<?php

namespace App\Controllers;

use App\Models\Receiving;
use App\Models\Supplier;
use Config\OSPOS;
use Config\Services;

class Suppliers extends Persons
{
    private array $config;

    private Receiving $receiving;

    private Supplier $supplier;

    public function __construct()
    {
        parent::__construct('suppliers');

        $this->config = config(OSPOS::class)->settings;
        $this->receiving = model(Receiving::class);
        $this->supplier = model(Supplier::class);
    }

    /**
     * @return void
     */
    public function getIndex(): void
    {
        $data['table_headers'] = get_suppliers_manage_table_headers();
        $data['controller_name'] = $this->global_view_data['controller_name'];
        $data['config'] = $this->global_view_data['config'];
        $data['employee_id'] = $this->global_view_data['user_info']->person_id;

        echo view('people/manage', $data);
    }

    /**
     * Gets one row for a supplier manage table. This is called using AJAX to update one row.
     * @param $row_id
     * @return void
     */
    public function getRow($row_id): void
    {
        $supplier = $this->supplier->get_info($row_id);
        $total_purchase = $this->supplier->get_total_purchase($row_id);
        $credit_stats = $this->supplier->get_credit_stats($row_id);
        $data_row = get_supplier_data_row($supplier, $total_purchase, $credit_stats);
        $data_row['category'] = $this->supplier->get_category_name($data_row['category']);

        echo json_encode($data_row);
    }

    /**
     * Returns Supplier table data rows. This will be called with AJAX.
     * @return void
     **/
    public function getSearch(): void
    {
        $search = $this->request->getGet('search');
        $limit = $this->request->getGet('limit', FILTER_SANITIZE_NUMBER_INT);
        $offset = $this->request->getGet('offset', FILTER_SANITIZE_NUMBER_INT);
        $sort = $this->sanitizeSortColumn(supplier_headers(), $this->request->getGet('sort', FILTER_SANITIZE_FULL_SPECIAL_CHARS), 'people.person_id');
        $order = $this->request->getGet('order', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

        $sortable_columns = ['people.person_id', 'company_name', 'category', 'email', 'phone_number'];
        if (!in_array($sort, $sortable_columns, true)) {
            $sort = 'people.person_id';
        }

        $suppliers = $this->supplier->search($search, $limit, $offset, $sort, $order);
        $total_rows = $this->supplier->get_found_rows($search);

        $data_rows = [];

        foreach ($suppliers->getResult() as $supplier) {
            $total_purchase = $this->supplier->get_total_purchase($supplier->person_id);
            $credit_stats = $this->supplier->get_credit_stats($supplier->person_id);
            $row = get_supplier_data_row($supplier, $total_purchase, $credit_stats);
            $row['category'] = $this->supplier->get_category_name($row['category']);
            $data_rows[] = $row;
        }

        echo json_encode(['total' => $total_rows, 'rows' => $data_rows]);
    }

    /**
     * Gives search suggestions based on what is being searched for
     **/
    public function getSuggest(): void
    {
        $search = $this->request->getGet('term');
        $suggestions = $this->supplier->get_search_suggestions($search, true);

        echo json_encode($suggestions);
    }

    /**
     * @return void
     */
    public function suggest_search(): void
    {
        $search = $this->request->getPost('term');
        $suggestions = $this->supplier->get_search_suggestions($search, false);

        echo json_encode($suggestions);
    }

    /**
     * Loads the supplier edit form
     *
     * @param int $supplier_id
     * @return void
     */
    public function getView(int $supplier_id = NEW_ENTRY): void
    {
        $info = $this->supplier->get_info($supplier_id);
        foreach (get_object_vars($info) as $property => $value) {
            $info->$property = $value;
        }
        $data['person_info'] = $info;
        $data['categories'] = $this->supplier->get_categories();

        echo view("suppliers/form", $data);
    }

    /**
     * Inserts/updates a supplier
     *
     * @param int $supplier_id
     * @return void
     */
    public function postSave(int $supplier_id = NEW_ENTRY): void
    {
        $first_name = $this->request->getPost('first_name', FILTER_SANITIZE_FULL_SPECIAL_CHARS);    // TODO: Duplicate code
        $last_name = $this->request->getPost('last_name', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $email = strtolower($this->request->getPost('email', FILTER_SANITIZE_EMAIL));

        // Format first and last name properly
        $first_name = $this->nameize($first_name);
        $last_name = $this->nameize($last_name);

        $person_data = [
            'first_name'   => $first_name,
            'last_name'    => $last_name,
            'gender'       => $this->request->getPost('gender'),
            'email'        => $email,
            'phone_number' => $this->request->getPost('phone_number', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'address_1'    => $this->request->getPost('address_1', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'address_2'    => $this->request->getPost('address_2', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'city'         => $this->request->getPost('city', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'state'        => $this->request->getPost('state', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'zip'          => $this->request->getPost('zip', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'country'      => $this->request->getPost('country', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'comments'     => $this->request->getPost('comments', FILTER_SANITIZE_FULL_SPECIAL_CHARS)
        ];

        $supplier_data = [
            'company_name'   => $this->request->getPost('company_name', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'agency_name'    => $this->request->getPost('agency_name', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'category'       => $this->request->getPost('category', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'account_number' => $this->request->getPost('account_number') == '' ? null : $this->request->getPost('account_number', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'tax_id'         => $this->request->getPost('tax_id', FILTER_SANITIZE_NUMBER_INT)
        ];

        if ($this->supplier->save_supplier($person_data, $supplier_data, $supplier_id)) {
            // New supplier
            if ($supplier_id == NEW_ENTRY) {
                echo json_encode([
                    'success' => true,
                    'message' => lang('Suppliers.successful_adding') . ' ' . $supplier_data['company_name'],
                    'id'      => $supplier_data['person_id']
                ]);
            } else { // Existing supplier

                echo json_encode([
                    'success' => true,
                    'message' => lang('Suppliers.successful_updating') . ' ' . $supplier_data['company_name'],
                    'id'      => $supplier_id
                ]);
            }
        } else { // Failure
            echo json_encode([
                'success' => false,
                'message' => lang('Suppliers.error_adding_updating') . ' ' .     $supplier_data['company_name'],
                'id'      => NEW_ENTRY
            ]);
        }
    }

    /**
     * Shows receivings for a supplier (Total Purchase link).
     */
    public function getReceivings(int $supplier_id): void
    {
        $info = $this->supplier->get_info($supplier_id);
        $receivings = $this->supplier->get_all_receivings($supplier_id);

        $data = [
            'supplier_name' => $info->company_name ?: ($info->first_name . ' ' . $info->last_name),
            'receivings'    => $receivings,
            'config'        => $this->config
        ];

        echo view('suppliers/receivings', $data);
    }

    /**
     * Shows credit receivings for a supplier.
     */
    public function getCredit_receivings(int $supplier_id): void
    {
        $info = $this->supplier->get_info($supplier_id);
        $credit_receivings = $this->supplier->get_credit_receivings($supplier_id);

        $data = [
            'supplier_name'    => $info->company_name ?: ($info->first_name . ' ' . $info->last_name),
            'credit_receivings' => $credit_receivings,
            'config'           => $this->config
        ];

        echo view('suppliers/credit_receivings', $data);
    }

    /**
     * Shows payments made to a supplier.
     */
    public function getPayments(int $supplier_id): void
    {
        $info = $this->supplier->get_info($supplier_id);
        $payments = $this->supplier->get_payments_by_supplier($supplier_id);

        $data = [
            'supplier_name' => $info->company_name ?: ($info->first_name . ' ' . $info->last_name),
            'payments'      => $payments,
            'config'        => $this->config
        ];

        echo view('suppliers/payments', $data);
    }

    /**
     * Shows Make Payment form for a supplier.
     */
    public function getReceive_payment(int $supplier_id): void
    {
        $info = $this->supplier->get_info($supplier_id);
        $credit_stats = $this->supplier->get_credit_stats($supplier_id);

        $payment_options = $this->receiving->get_payment_options();
        unset($payment_options[lang('Sales.due')]);

        $data = [
            'supplier_id'     => $supplier_id,
            'supplier_name'   => $info->company_name ?: ($info->first_name . ' ' . $info->last_name),
            'credit_balance'  => $credit_stats->credit_balance,
            'payment_options' => $payment_options,
            'config'          => $this->config
        ];

        echo view('suppliers/form_receive_payment', $data);
    }

    /**
     * Processes Make Payment - applies amount to oldest credit receivings first (FIFO).
     */
    public function postReceive_payment(): void
    {
        $supplier_id = $this->request->getPost('supplier_id', FILTER_SANITIZE_NUMBER_INT);
        $amount = parse_decimals($this->request->getPost('payment_amount'));
        $payment_type = $this->request->getPost('payment_type', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

        if (empty($supplier_id) || $amount <= 0 || empty($payment_type)) {
            echo json_encode(['success' => false, 'message' => lang('Common.fields_required_message')]);
            return;
        }

        $credit_receivings = $this->supplier->get_credit_receivings_for_fifo($supplier_id);
        if (empty($credit_receivings)) {
            echo json_encode(['success' => false, 'message' => lang('Suppliers.no_credit_receivings')]);
            return;
        }

        $remaining = $amount;
        $employee = model(\App\Models\Employee::class);
        $employee_id = $employee->get_logged_in_employee_info()->person_id;

        $db = \Config\Database::connect();
        $db->transStart();

        foreach ($credit_receivings as $recv) {
            if ($remaining <= 0) {
                break;
            }
            $apply_amount = min($remaining, $recv->balance);
            if ($apply_amount <= 0) {
                continue;
            }

            $builder = $db->table('receivings_payments');
            $builder->insert([
                'receiving_id'   => $recv->receiving_id,
                'payment_type'   => $payment_type,
                'payment_amount' => $apply_amount,
                'employee_id'    => $employee_id
            ]);

            $remaining -= $apply_amount;
        }

        $db->transComplete();

        if ($db->transStatus()) {
            \Config\Services::cache()->delete('dashboard_credit_purchase');
            $info = $this->supplier->get_info($supplier_id);
            echo json_encode([
                'success' => true,
                'message' => lang('Suppliers.payment_made') . ' ' . to_currency($amount),
                'id'      => $supplier_id
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => lang('Suppliers.payment_failed')]);
        }
    }

    /**
     * This deletes suppliers from the suppliers table
     *
     * @return void
     */
    public function postDelete(): void
    {
        $suppliers_to_delete = $this->request->getPost('ids', FILTER_SANITIZE_NUMBER_INT);

        if ($this->supplier->delete_list($suppliers_to_delete)) {
            echo json_encode([
                'success' => true,
                'message' => lang('Suppliers.successful_deleted') . ' ' . count($suppliers_to_delete) . ' ' . lang('Suppliers.one_or_multiple')
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => lang('Suppliers.cannot_be_deleted')]);
        }
    }
}
