<?php

namespace App\Database\Migrations;

use CodeIgniter\Database\Migration;

class Migration_branch_transfers extends Migration
{
    /**
     * Create branch_transfers and branch_transfer_items tables for inter-branch transfers.
     * Add transfers module, permission, and grant.
     */
    public function up(): void
    {
        $prefix = $this->db->getPrefix();

        $this->db->query("CREATE TABLE IF NOT EXISTS `{$prefix}branch_transfers` (
            `transfer_id` int(11) NOT NULL AUTO_INCREMENT,
            `from_location_id` int(11) NOT NULL,
            `to_location_id` int(11) NOT NULL,
            `transfer_time` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `employee_id` int(11) NOT NULL,
            `comment` text,
            PRIMARY KEY (`transfer_id`),
            KEY `from_location_id` (`from_location_id`),
            KEY `to_location_id` (`to_location_id`),
            KEY `employee_id` (`employee_id`),
            CONSTRAINT `branch_transfers_ibfk_1` FOREIGN KEY (`from_location_id`) REFERENCES `{$prefix}stock_locations` (`location_id`),
            CONSTRAINT `branch_transfers_ibfk_2` FOREIGN KEY (`to_location_id`) REFERENCES `{$prefix}stock_locations` (`location_id`),
            CONSTRAINT `branch_transfers_ibfk_3` FOREIGN KEY (`employee_id`) REFERENCES `{$prefix}employees` (`person_id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8");

        $this->db->query("CREATE TABLE IF NOT EXISTS `{$prefix}branch_transfer_items` (
            `transfer_id` int(11) NOT NULL,
            `item_id` int(11) NOT NULL,
            `quantity` decimal(15,3) NOT NULL DEFAULT 0,
            PRIMARY KEY (`transfer_id`, `item_id`),
            KEY `item_id` (`item_id`),
            CONSTRAINT `branch_transfer_items_ibfk_1` FOREIGN KEY (`transfer_id`) REFERENCES `{$prefix}branch_transfers` (`transfer_id`) ON DELETE CASCADE,
            CONSTRAINT `branch_transfer_items_ibfk_2` FOREIGN KEY (`item_id`) REFERENCES `{$prefix}items` (`item_id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8");

        $moduleMissing = $this->db->table($prefix . 'modules')->where('module_id', 'transfers')->countAllResults() === 0;
        if ($moduleMissing) {
            $this->db->query("INSERT INTO `{$prefix}modules` (`name_lang_key`, `desc_lang_key`, `sort`, `module_id`) VALUES
                ('module_transfers', 'module_transfers_desc', 55, 'transfers')");
        }

        $permExists = $this->db->table($prefix . 'permissions')->where('permission_id', 'transfers')->countAllResults() === 0;
        if ($permExists) {
            $this->db->query("INSERT INTO `{$prefix}permissions` (`permission_id`, `module_id`) VALUES ('transfers', 'transfers')");
        }

        $grantsTable = $prefix . 'grants';
        $grantExists = $this->db->table($grantsTable)->where('permission_id', 'transfers')->countAllResults() > 0;
        if (!$grantExists) {
            $hasMenuGroup = $this->db->fieldExists('menu_group', $grantsTable);
            $peopleWithItems = $this->db->table($grantsTable)->select('person_id')->where('permission_id', 'items')->get()->getResultArray();
            $personIds = empty($peopleWithItems) ? [1] : array_unique(array_column($peopleWithItems, 'person_id'));
            foreach ($personIds as $pid) {
                $pid = (int) $pid;
                $menuGroup = $hasMenuGroup ? 'both' : null;
                $data = ['permission_id' => 'transfers', 'person_id' => $pid];
                if ($menuGroup !== null) {
                    $data['menu_group'] = $menuGroup;
                }
                $this->db->table($grantsTable)->insert($data);
            }
        }

    }

    public function down(): void
    {
        $prefix = $this->db->getPrefix();

        $this->db->query('DROP TABLE IF EXISTS ' . $this->db->prefixTable('branch_transfer_items'));
        $this->db->query('DROP TABLE IF EXISTS ' . $this->db->prefixTable('branch_transfers'));

        $this->db->table('grants')->where('permission_id', 'transfers')->delete();
        $this->db->table('permissions')->where('permission_id', 'transfers')->delete();
        $this->db->table('modules')->where('module_id', 'transfers')->delete();
    }
}
