<?php

namespace App\Database\Migrations;

use CodeIgniter\Database\Migration;

class Migration_add_carton_pieces_to_sales_receivings extends Migration
{
    /**
     * Add carton_qty and pieces_qty to sales_items and receivings_items.
     * Enables storing the carton/pieces breakdown for reports and receipts.
     */
    public function up(): void
    {
        helper('locale');
        $prefix = $this->db->getPrefix() ?: '';

        $salesTable = $prefix . 'sales_items';
        if ($this->db->tableExists($salesTable) && !$this->db->fieldExists('carton_qty', $salesTable)) {
            $this->db->query("ALTER TABLE `{$salesTable}` ADD COLUMN `carton_qty` decimal(15,3) NOT NULL DEFAULT 0 AFTER `quantity_purchased`");
            $this->db->query("ALTER TABLE `{$salesTable}` ADD COLUMN `pieces_qty` decimal(15,3) NOT NULL DEFAULT 0 AFTER `carton_qty`");
            $this->backfill_sales_items($salesTable, $prefix);
        }

        $recvTable = $prefix . 'receivings_items';
        if ($this->db->tableExists($recvTable) && !$this->db->fieldExists('carton_qty', $recvTable)) {
            $this->db->query("ALTER TABLE `{$recvTable}` ADD COLUMN `carton_qty` decimal(15,3) NOT NULL DEFAULT 0 AFTER `receiving_quantity`");
            $this->db->query("ALTER TABLE `{$recvTable}` ADD COLUMN `pieces_qty` decimal(15,3) NOT NULL DEFAULT 0 AFTER `carton_qty`");
            $this->backfill_receivings_items($recvTable, $prefix);
        }
    }

    public function down(): void
    {
        $prefix = $this->db->getPrefix() ?: '';

        $salesTable = $prefix . 'sales_items';
        if ($this->db->fieldExists('carton_qty', $salesTable)) {
            $this->db->query("ALTER TABLE `{$salesTable}` DROP COLUMN `carton_qty`, DROP COLUMN `pieces_qty`");
        }

        $recvTable = $prefix . 'receivings_items';
        if ($this->db->fieldExists('carton_qty', $recvTable)) {
            $this->db->query("ALTER TABLE `{$recvTable}` DROP COLUMN `carton_qty`, DROP COLUMN `pieces_qty`");
        }
    }

    private function backfill_sales_items(string $table, string $prefix): void
    {
        $qtyCol = $this->getQuantityColumn($prefix . 'sales_items');
        $itemsTable = $prefix . 'items';
        $hasItemUnit = $this->db->fieldExists('item_unit', $itemsTable);

        $sql = "SELECT si.sale_id, si.item_id, si.line, si.{$qtyCol} AS qty, i.item_unit, i.qty_per_unit, i.qty_per_carton, i.qty_per_box, i.qty_per_bag, i.qty_per_pack, i.pack_name
            FROM {$table} si
            LEFT JOIN {$itemsTable} i ON si.item_id = i.item_id";
        $rows = $this->db->query($sql)->getResultArray();

        foreach ($rows as $row) {
            $item = $hasItemUnit ? (object) $row : null;
            $qp = quantity_to_carton_pieces((float) $row['qty'], $item);
            $carton = $qp['carton'];
            $pieces = $qp['pieces'];
            $saleId = (int) $row['sale_id'];
            $itemId = (int) $row['item_id'];
            $line = (int) $row['line'];
            $this->db->query("UPDATE `{$table}` SET carton_qty = {$this->db->escape($carton)}, pieces_qty = {$this->db->escape($pieces)} WHERE sale_id = {$saleId} AND item_id = {$itemId} AND line = {$line}");
        }
    }

    private function backfill_receivings_items(string $table, string $prefix): void
    {
        $itemsTable = $prefix . 'items';
        $hasItemUnit = $this->db->fieldExists('item_unit', $itemsTable);

        $sql = "SELECT ri.receiving_id, ri.item_id, ri.line, (ri.quantity_purchased * ri.receiving_quantity) AS total_pieces,
            i.item_unit, i.qty_per_unit, i.qty_per_carton, i.qty_per_box, i.qty_per_bag, i.qty_per_pack, i.pack_name
            FROM {$table} ri
            LEFT JOIN {$itemsTable} i ON ri.item_id = i.item_id";
        $rows = $this->db->query($sql)->getResultArray();

        foreach ($rows as $row) {
            $item = $hasItemUnit ? (object) $row : null;
            $qp = quantity_to_carton_pieces(abs((float) $row['total_pieces']), $item);
            $carton = $qp['carton'];
            $pieces = $qp['pieces'];
            $recvId = (int) $row['receiving_id'];
            $itemId = (int) $row['item_id'];
            $line = (int) $row['line'];
            $this->db->query("UPDATE `{$table}` SET carton_qty = {$this->db->escape($carton)}, pieces_qty = {$this->db->escape($pieces)} WHERE receiving_id = {$recvId} AND item_id = {$itemId} AND line = {$line}");
        }
    }

    private function getQuantityColumn(string $table): string
    {
        $candidates = ['quantity', 'quantity_purchased', 'qty'];
        $escaped = $this->db->escape($table);
        $escapedCandidates = array_map([$this->db, 'escape'], $candidates);
        $inList = implode(',', $escapedCandidates);
        $sql = "SELECT COLUMN_NAME FROM information_schema.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = {$escaped}
            AND COLUMN_NAME IN ({$inList})
            ORDER BY FIELD(COLUMN_NAME, {$inList})
            LIMIT 1";
        $result = $this->db->query($sql);
        if ($result !== false && ($row = $result->getRow())) {
            $arr = (array) $row;
            return $arr['COLUMN_NAME'] ?? $arr['column_name'] ?? 'quantity_purchased';
        }
        return 'quantity_purchased';
    }
}
