<?php

namespace App\Models;

use CodeIgniter\Database\ResultInterface;
use CodeIgniter\Model;

/**
 * Branch_transfer model - inter-branch inventory transfers.
 */
class Branch_transfer extends Model
{
    protected $table = 'branch_transfers';
    protected $primaryKey = 'transfer_id';
    protected $useAutoIncrement = true;
    protected $useSoftDeletes = false;
    protected $allowedFields = [
        'from_location_id',
        'to_location_id',
        'transfer_time',
        'employee_id',
        'comment'
    ];

    /**
     * Create a new transfer and update inventory.
     *
     * @param int $fromLocationId Source branch
     * @param int $toLocationId Destination branch
     * @param array $items [['item_id' => int, 'quantity' => float], ...]
     * @param int $employeeId
     * @param string|null $comment
     * @return int|false Transfer ID or false on failure
     */
    public function create_transfer(int $fromLocationId, int $toLocationId, array $items, int $employeeId, ?string $comment = null)
    {
        if ($fromLocationId === $toLocationId) {
            return false;
        }
        if (empty($items)) {
            return false;
        }

        $itemQuantity = model(Item_quantity::class);
        $inventory = model(Inventory::class);
        $stockLocation = model(Stock_location::class);

        $this->db->transStart();

        $this->insert([
            'from_location_id' => $fromLocationId,
            'to_location_id'   => $toLocationId,
            'employee_id'      => $employeeId,
            'comment'          => $comment ?? ''
        ]);
        $transferId = $this->getInsertID();
        if (!$transferId) {
            $this->db->transComplete();
            return false;
        }

        $fromName = $stockLocation->get_location_name($fromLocationId);
        $toName = $stockLocation->get_location_name($toLocationId);

        foreach ($items as $row) {
            $itemId = (int) ($row['item_id'] ?? 0);
            $quantity = (float) ($row['quantity'] ?? 0);
            if ($itemId <= 0 || $quantity <= 0) {
                continue;
            }

            $currentQty = $itemQuantity->get_item_quantity($itemId, $fromLocationId);
            $available = (float) ($currentQty->quantity ?? 0);
            if ($available < $quantity) {
                $this->db->transComplete();
                return false;
            }

            $this->db->table('branch_transfer_items')->insert([
                'transfer_id' => $transferId,
                'item_id'    => $itemId,
                'quantity'   => $quantity
            ]);

            $itemQuantity->change_quantity($itemId, $fromLocationId, -$quantity);
            $itemQuantity->change_quantity($itemId, $toLocationId, $quantity);

            $invComment = lang('Transfers.transfer_out', [$quantity, $toName]);
            $inventory->insert([
                'trans_items'     => $itemId,
                'trans_user'      => $employeeId,
                'trans_comment'   => $invComment,
                'trans_location'  => $fromLocationId,
                'trans_inventory' => -$quantity
            ], false);

            $invCommentIn = lang('Transfers.transfer_in', [$quantity, $fromName]);
            $inventory->insert([
                'trans_items'     => $itemId,
                'trans_user'      => $employeeId,
                'trans_comment'   => $invCommentIn,
                'trans_location'  => $toLocationId,
                'trans_inventory' => $quantity
            ], false);
        }

        $this->db->transComplete();

        return $this->db->transStatus() ? $transferId : false;
    }

    /**
     * Get transfer with items.
     */
    public function get_transfer_with_items(int $transferId): ?object
    {
        $row = $this->db->table('branch_transfers AS bt')
            ->select('bt.*, 
                fl.location_name AS from_location_name, 
                tl.location_name AS to_location_name,
                CONCAT(ep.first_name, " ", ep.last_name) AS employee_name')
            ->join('stock_locations AS fl', 'fl.location_id = bt.from_location_id')
            ->join('stock_locations AS tl', 'tl.location_id = bt.to_location_id')
            ->join('employees AS e', 'e.person_id = bt.employee_id')
            ->join('people AS ep', 'ep.person_id = e.person_id')
            ->where('bt.transfer_id', $transferId)
            ->get()
            ->getRow();
        if (!$row) {
            return null;
        }
        $itemsTable = $this->db->getPrefix() . 'items';
        $selectFields = 'branch_transfer_items.item_id, branch_transfer_items.quantity, items.name, items.item_number';
        if ($this->db->fieldExists('qty_per_carton', $itemsTable)) {
            $selectFields .= ', items.qty_per_carton, items.qty_per_box, items.qty_per_pack, items.qty_per_bag, items.item_unit, items.pack_name';
        }
        $items = $this->db->table('branch_transfer_items')
            ->select($selectFields)
            ->join('items', 'items.item_id = branch_transfer_items.item_id')
            ->where('transfer_id', $transferId)
            ->get()
            ->getResult();
        $row->items = $items;
        return $row;
    }

    /**
     * Get report data for transfers in date range.
     *
     * @param string $startDate Y-m-d
     * @param string $endDate Y-m-d
     * @param string $locationId 'all' or location_id to filter by from_location or to_location
     * @return array ['summary' => [...], 'details' => [transfer_id => [...]]]
     */
    public function get_report_data(string $startDate, string $endDate, string $locationId = 'all'): array
    {
        $builder = $this->db->table('branch_transfers AS bt');
        $builder->select('bt.transfer_id, bt.transfer_time,
            fl.location_name AS from_location_name,
            tl.location_name AS to_location_name,
            CONCAT(ep.first_name, " ", ep.last_name) AS employee_name,
            bt.comment');
        $builder->join('stock_locations AS fl', 'fl.location_id = bt.from_location_id');
        $builder->join('stock_locations AS tl', 'tl.location_id = bt.to_location_id');
        $builder->join('employees AS e', 'e.person_id = bt.employee_id');
        $builder->join('people AS ep', 'ep.person_id = e.person_id');
        $builder->where('DATE(bt.transfer_time) >=', $startDate);
        $builder->where('DATE(bt.transfer_time) <=', $endDate);
        if ($locationId !== 'all' && $locationId !== '') {
            $locId = (int) $locationId;
            $builder->groupStart()
                ->where('bt.from_location_id', $locId)
                ->orWhere('bt.to_location_id', $locId)
                ->groupEnd();
        }
        $builder->orderBy('bt.transfer_time', 'ASC');
        $rows = $builder->get()->getResultArray();

        $summary = [];
        $details = [];
        foreach ($rows as $row) {
            $summary[] = $row;
            $select = 'bti.item_id, bti.quantity, i.item_number, i.name, i.category';
            if ($this->db->fieldExists('item_unit', $this->db->getPrefix() . 'items')) {
                $select .= ', i.item_unit, i.qty_per_unit, i.qty_per_carton, i.qty_per_box, i.qty_per_bag, i.qty_per_pack, i.pack_name';
            }
            $items = $this->db->table('branch_transfer_items AS bti')
                ->select($select)
                ->join('items AS i', 'i.item_id = bti.item_id')
                ->where('bti.transfer_id', $row['transfer_id'])
                ->get()
                ->getResultArray();
            $details[$row['transfer_id']] = $items;
        }

        return ['summary' => $summary, 'details' => $details];
    }

    /**
     * Get all transfers with pagination.
     */
    public function get_all(int $limit = 50, int $offset = 0): ResultInterface
    {
        $builder = $this->db->table('branch_transfers AS bt');
        $builder->select('bt.*, 
            fl.location_name AS from_location_name, 
            tl.location_name AS to_location_name,
            CONCAT(ep.first_name, " ", ep.last_name) AS employee_name');
        $builder->join('stock_locations AS fl', 'fl.location_id = bt.from_location_id');
        $builder->join('stock_locations AS tl', 'tl.location_id = bt.to_location_id');
        $builder->join('employees AS e', 'e.person_id = bt.employee_id');
        $builder->join('people AS ep', 'ep.person_id = e.person_id');
        $builder->orderBy('bt.transfer_time', 'DESC');
        $builder->limit($limit, $offset);

        return $builder->get();
    }
}
