<?php

namespace App\Models;

use CodeIgniter\Database\ResultInterface;
use CodeIgniter\Model;
use Config\OSPOS;
use ReflectionException;

/**
 * Receiving class
 */
class Receiving extends Model
{
    protected $table = 'receivings';
    protected $primaryKey = 'receiving_id';
    protected $useAutoIncrement = true;
    protected $useSoftDeletes = false;
    protected $allowedFields = [
        'receiving_time',
        'supplier_id',
        'employee_id',
        'comment',
        'receiving_id',
        'payment_type',
        'reference'
    ];

    /**
     * Compute total pieces from a cart item for inventory. Handles bulk (qty_carton + qty_piece) and standard items.
     *
     * @param array $item_data Cart item
     * @return float Total pieces for inventory
     */
    protected function get_items_received_from_cart_item(array $item_data): float
    {
        $rq = (float) ($item_data['receiving_quantity'] ?? 1);
        if (isset($item_data['qty_carton'], $item_data['qty_piece']) && $rq > 0) {
            $cartons = (float) ($item_data['qty_carton'] ?? 0);
            $pieces = (float) ($item_data['qty_piece'] ?? 0);

            return $cartons * $rq + $pieces;
        }
        $qty = (float) ($item_data['quantity'] ?? 0);

        return $rq != 0 ? $qty * $rq : $qty;
    }

    /**
     * Get quantity_purchased and receiving_quantity for receivings_items so that their product = items_received.
     * Preserves carton/piece breakdown when possible for reload; otherwise stores total pieces.
     *
     * @param array $item_data Cart item
     * @param float $items_received Total pieces
     * @return array [quantity_purchased, receiving_quantity]
     */
    protected function get_receivings_item_quantities(array $item_data, float $items_received): array
    {
        $rq = (float) ($item_data['receiving_quantity'] ?? 1);
        if (isset($item_data['qty_carton'], $item_data['qty_piece']) && $rq > 0) {
            $pieces = (float) ($item_data['qty_piece'] ?? 0);
            if ($pieces == 0) {
                return [(float) ($item_data['qty_carton'] ?? 0), $rq];
            }
        }
        $qty = (float) ($item_data['quantity'] ?? 0);
        if ($rq > 0 && abs(($qty * $rq) - $items_received) < 0.001) {
            return [$qty, $rq];
        }

        return [$items_received, 1];
    }

    /**
     * @param int $receiving_id
     * @return ResultInterface
     */
    public function get_info(int $receiving_id): ResultInterface
    {
        $builder = $this->db->table('receivings');
        $builder->join('people', 'people.person_id = receivings.supplier_id', 'LEFT');
        $builder->join('suppliers', 'suppliers.person_id = receivings.supplier_id', 'LEFT');
        $builder->where('receiving_id', $receiving_id);

        return $builder->get();
    }

    /**
     * @param string $reference
     * @return ResultInterface
     */
    public function get_receiving_by_reference(string $reference): ResultInterface
    {
        $builder = $this->db->table('receivings');
        $builder->where('reference', $reference);

        return $builder->get();
    }

    /**
     * @param string $receipt_receiving_id
     * @return bool
     */
    public function is_valid_receipt(string $receipt_receiving_id): bool    // TODO: maybe receipt_receiving_id should be an array rather than a space delimited string
    {
        if (!empty($receipt_receiving_id)) {
            // RECV #
            $pieces = explode(' ', $receipt_receiving_id);

            if (count($pieces) == 2 && preg_match('/(RECV|KIT)/', $pieces[0])) {
                return $this->exists($pieces[1]);
            } else {
                return $this->get_receiving_by_reference($receipt_receiving_id)->getNumRows() > 0;
            }
        }

        return false;
    }

    /**
     * @param int $receiving_id
     * @return bool
     */
    public function exists(int $receiving_id): bool
    {
        $builder = $this->db->table('receivings');
        $builder->where('receiving_id', $receiving_id);

        return ($builder->get()->getNumRows() == 1);
    }

    /**
     * @param $receiving_id
     * @param $receiving_data
     * @return bool
     */
    public function update($receiving_id = null, $receiving_data = null): bool
    {
        $builder = $this->db->table('receivings');
        $builder->where('receiving_id', $receiving_id);

        return $builder->update($receiving_data);
    }

    /**
     * @throws ReflectionException
     */
    public function save_value(array $items, int $supplier_id, int $employee_id, string $comment, string $reference, ?string $payment_type, int $receiving_id = NEW_ENTRY): int    // TODO: $receiving_id gets overwritten before it's evaluated. It doesn't make sense to pass this here.
    {
        $attribute = model(Attribute::class);
        $inventory = model('Inventory');
        $item = model(Item::class);
        $item_quantity = model(Item_quantity::class);
        $supplier = model(Supplier::class);

        if (count($items) == 0) {
            return -1;    // TODO: Replace -1 with a constant
        }

        $receivings_data = [
            'receiving_time' => date('Y-m-d H:i:s'),
            'supplier_id'    => $supplier->exists($supplier_id) ? $supplier_id : null,
            'employee_id'    => $employee_id,
            'payment_type'   => $payment_type,
            'comment'        => $comment,
            'reference'      => $reference
        ];

        // Run these queries as a transaction, we want to make sure we do all or nothing
        $this->db->transStart();

        $builder = $this->db->table('receivings');
        $builder->insert($receivings_data);
        $receiving_id = $this->db->insertID();

        $builder = $this->db->table('receivings_items');

        foreach ($items as $line => $item_data) {
            $config = config(OSPOS::class)->settings;
            $cur_item_info = $item->get_info($item_data['item_id']);
            $items_received = $this->get_items_received_from_cart_item($item_data);

            [$qty_purchased, $recv_qty] = $this->get_receivings_item_quantities($item_data, $items_received);

            // carton_qty and pieces_qty from cart or derived from items_received
            if (isset($item_data['qty_carton'], $item_data['qty_piece'])) {
                $carton_qty = (float) $item_data['qty_carton'];
                $pieces_qty = (float) $item_data['qty_piece'];
            } else {
                $qp = quantity_to_carton_pieces($items_received, $cur_item_info);
                $carton_qty = $qp['has_bulk'] ? (float) $qp['carton'] : 0.0;
                $pieces_qty = (float) $qp['pieces'];
            }

            $receivings_items_data = [
                'receiving_id'       => $receiving_id,
                'item_id'            => $item_data['item_id'],
                'line'               => $item_data['line'],
                'description'        => $item_data['description'],
                'serialnumber'       => $item_data['serialnumber'],
                'quantity_purchased' => $qty_purchased,
                'receiving_quantity' => $recv_qty,
                'carton_qty'         => $carton_qty,
                'pieces_qty'         => $pieces_qty,
                'discount'           => $item_data['discount'],
                'discount_type'      => $item_data['discount_type'],
                'item_cost_price'    => $cur_item_info->cost_price,
                'item_unit_price'    => $item_data['price'],
                'item_location'      => $item_data['item_location']
            ];
            if ($this->db->fieldExists('expire_date', $this->db->getPrefix() . 'receivings_items')) {
                $receivings_items_data['expire_date'] = !empty($item_data['expire_date']) ? $item_data['expire_date'] : null;
            }

            $builder->insert($receivings_items_data);

            // Update cost price, if changed AND is set in config as wanted
            if ($cur_item_info->cost_price != $item_data['price'] && $config['receiving_calculate_average_price']) {
                $item->change_cost_price($item_data['item_id'], $items_received, $item_data['price'], $cur_item_info->cost_price);
            }

            // Update stock quantity
            $item_quantity_value = $item_quantity->get_item_quantity($item_data['item_id'], $item_data['item_location']);
            $item_quantity->save_value(
                [
                    'quantity'    => $item_quantity_value->quantity + $items_received,
                    'item_id'     => $item_data['item_id'],
                    'location_id' => $item_data['item_location']
                ],
                $item_data['item_id'],
                $item_data['item_location']
            );

            $recv_remarks = 'RECV ' . $receiving_id;
            $inv_data = [
                'trans_date'      => date('Y-m-d H:i:s'),
                'trans_items'     => $item_data['item_id'],
                'trans_user'      => $employee_id,
                'trans_location'  => $item_data['item_location'],
                'trans_comment'   => $recv_remarks,
                'trans_inventory' => $items_received
            ];

            $inventory->insert($inv_data, false);
            $attribute->copy_attribute_links($item_data['item_id'], 'receiving_id', $receiving_id);
        }

        $this->db->transComplete();

        return $this->db->transStatus() ? $receiving_id : -1;
    }

    /**
     * Update an existing receiving with new items. Reverts inventory for old items, deletes old lines,
     * updates the receivings row, and inserts new items with inventory updates.
     *
     * @throws ReflectionException
     */
    public function update_value(int $receiving_id, array $items, int $supplier_id, int $employee_id, string $comment, string $reference, ?string $payment_type): bool
    {
        $attribute = model(Attribute::class);
        $inventory = model('Inventory');
        $item = model(Item::class);
        $item_quantity = model(Item_quantity::class);
        $supplier = model(Supplier::class);

        if (count($items) == 0) {
            return false;
        }

        $this->db->transStart();

        // Revert inventory for existing items
        $existing_items = $this->get_receiving_items($receiving_id)->getResultArray();
        foreach ($existing_items as $row) {
            $rq = (float) ($row['receiving_quantity'] ?? 1);
            $qp = (float) ($row['quantity_purchased'] ?? 0);
            $items_received = $rq > 0 ? $qp * $rq : $qp;
            $revert_qty = -$items_received;

            $inv_data = [
                'trans_date'      => date('Y-m-d H:i:s'),
                'trans_items'     => $row['item_id'],
                'trans_user'      => $employee_id,
                'trans_comment'   => 'Revert receiving ' . $receiving_id . ' for edit',
                'trans_location'  => $row['item_location'],
                'trans_inventory' => $revert_qty
            ];
            $inventory->insert($inv_data, false);
            $item_quantity->change_quantity($row['item_id'], $row['item_location'], $revert_qty);
        }

        // Delete existing receivings_items
        $builder = $this->db->table('receivings_items');
        $builder->where('receiving_id', $receiving_id);
        $builder->delete();

        // Update receivings header
        $receivings_data = [
            'receiving_time' => date('Y-m-d H:i:s'),
            'supplier_id'    => $supplier->exists($supplier_id) ? $supplier_id : null,
            'employee_id'    => $employee_id,
            'payment_type'   => $payment_type,
            'comment'        => $comment,
            'reference'      => $reference
        ];
        $builder = $this->db->table('receivings');
        $builder->where('receiving_id', $receiving_id);
        $builder->update($receivings_data);

        // Insert new items (same logic as save_value)
        $builder = $this->db->table('receivings_items');
        $config = config(OSPOS::class)->settings;

        foreach ($items as $line => $item_data) {
            $cur_item_info = $item->get_info($item_data['item_id']);
            $items_received = $this->get_items_received_from_cart_item($item_data);
            [$qty_purchased, $recv_qty] = $this->get_receivings_item_quantities($item_data, $items_received);

            // carton_qty and pieces_qty from cart or derived from items_received
            if (isset($item_data['qty_carton'], $item_data['qty_piece'])) {
                $carton_qty = (float) $item_data['qty_carton'];
                $pieces_qty = (float) $item_data['qty_piece'];
            } else {
                $qp = quantity_to_carton_pieces($items_received, $cur_item_info);
                $carton_qty = $qp['has_bulk'] ? (float) $qp['carton'] : 0.0;
                $pieces_qty = (float) $qp['pieces'];
            }

            $receivings_items_data = [
                'receiving_id'       => $receiving_id,
                'item_id'            => $item_data['item_id'],
                'line'               => $item_data['line'],
                'description'        => $item_data['description'],
                'serialnumber'       => $item_data['serialnumber'],
                'quantity_purchased' => $qty_purchased,
                'receiving_quantity' => $recv_qty,
                'carton_qty'         => $carton_qty,
                'pieces_qty'         => $pieces_qty,
                'discount'           => $item_data['discount'],
                'discount_type'      => $item_data['discount_type'],
                'item_cost_price'    => $cur_item_info->cost_price,
                'item_unit_price'    => $item_data['price'],
                'item_location'      => $item_data['item_location']
            ];
            if ($this->db->fieldExists('expire_date', $this->db->getPrefix() . 'receivings_items')) {
                $receivings_items_data['expire_date'] = !empty($item_data['expire_date']) ? $item_data['expire_date'] : null;
            }

            $builder->insert($receivings_items_data);

            if ($cur_item_info->cost_price != $item_data['price'] && $config['receiving_calculate_average_price']) {
                $item->change_cost_price($item_data['item_id'], $items_received, $item_data['price'], $cur_item_info->cost_price);
            }

            $item_quantity_value = $item_quantity->get_item_quantity($item_data['item_id'], $item_data['item_location']);
            $item_quantity->save_value(
                [
                    'quantity'    => $item_quantity_value->quantity + $items_received,
                    'item_id'     => $item_data['item_id'],
                    'location_id' => $item_data['item_location']
                ],
                $item_data['item_id'],
                $item_data['item_location']
            );

            $recv_remarks = 'RECV ' . $receiving_id;
            $inv_data = [
                'trans_date'      => date('Y-m-d H:i:s'),
                'trans_items'     => $item_data['item_id'],
                'trans_user'      => $employee_id,
                'trans_location'  => $item_data['item_location'],
                'trans_comment'   => $recv_remarks,
                'trans_inventory' => $items_received
            ];

            $inventory->insert($inv_data, false);
            $attribute->copy_attribute_links($item_data['item_id'], 'receiving_id', $receiving_id);
        }

        $this->db->transComplete();

        return $this->db->transStatus();
    }

    /**
     * @throws ReflectionException
     */
    public function delete_list(array $receiving_ids, int $employee_id, bool $update_inventory = true): bool
    {
        $success = true;

        // Start a transaction to assure data integrity
        $this->db->transStart();

        foreach ($receiving_ids as $receiving_id) {
            $success &= $this->delete_value($receiving_id, $employee_id, $update_inventory);
        }

        // Execute transaction
        $this->db->transComplete();

        $success &= $this->db->transStatus();

        return $success;
    }

    /**
     * @throws ReflectionException
     */
    public function delete_value(int $receiving_id, int $employee_id, bool $update_inventory = true): bool
    {
        // Start a transaction to assure data integrity
        $this->db->transStart();

        if ($update_inventory) {
            // TODO: defect, not all item deletions will be undone? get array with all the items involved in the sale to update the inventory tracking
            $items = $this->get_receiving_items($receiving_id)->getResultArray();

            $inventory = model('Inventory');
            $item_quantity = model(Item_quantity::class);

            foreach ($items as $item) {
                // Create query to update inventory tracking
                $inv_data = [
                    'trans_date'      => date('Y-m-d H:i:s'),
                    'trans_items'     => $item['item_id'],
                    'trans_user'      => $employee_id,
                    'trans_comment'   => 'Deleting receiving ' . $receiving_id,
                    'trans_location'  => $item['item_location'],
                    'trans_inventory' => $item['quantity_purchased'] * (-$item['receiving_quantity'])
                ];
                // Update inventory
                $inventory->insert($inv_data, false);

                // Update quantities
                $item_quantity->change_quantity($item['item_id'], $item['item_location'], $item['quantity_purchased'] * (-$item['receiving_quantity']));
            }
        }

        // Delete all items
        $builder = $this->db->table('receivings_items');
        $builder->delete(['receiving_id' => $receiving_id]);

        // Delete sale itself
        $builder = $this->db->table('receivings');
        $builder->delete(['receiving_id' => $receiving_id]);

        // Execute transaction
        $this->db->transComplete();

        return $this->db->transStatus();
    }

    /**
     * @param int $receiving_id
     * @return ResultInterface
     */
    public function get_receiving_items(int $receiving_id): ResultInterface
    {
        $builder = $this->db->table('receivings_items');
        $builder->where('receiving_id', $receiving_id);

        return $builder->get();
    }

    /**
     * Get received batches for an item (from receivings_items) with expire dates.
     * Used to show batch-level expire date info in item count details.
     *
     * @param int $item_id
     * @param int|null $location_id Optional location filter
     * @return array
     */
    public function get_received_batches_for_item(int $item_id, ?int $location_id = null): array
    {
        try {
            return $this->_get_received_batches_for_item($item_id, $location_id);
        } catch (\CodeIgniter\Database\Exceptions\DatabaseException $e) {
            if (strpos($e->getMessage(), 'Unknown column') !== false) {
                return $this->_get_received_batches_for_item($item_id, $location_id, true);
            }
            throw $e;
        }
    }

    /**
     * Internal: fetch received batches, with optional minimal schema (no quantity columns).
     */
    private function _get_received_batches_for_item(int $item_id, ?int $location_id, bool $minimal = false): array
    {
        $builder = $this->db->table('receivings_items');
        $select = [
            'receivings_items.receiving_id',
            'receivings.receiving_time',
            'receivings_items.line',
            'receivings_items.item_location',
        ];
        if ($minimal) {
            $select[] = '1 AS pieces_received';
        } else {
            $select[] = 'receivings_items.quantity_purchased';
            $select[] = 'receivings_items.receiving_quantity';
            $select[] = '(receivings_items.quantity_purchased * receivings_items.receiving_quantity) AS pieces_received';
        }
        $builder->select($select);
        if (!$minimal) {
            $table = $this->db->prefixTable('receivings_items');
            $columns = $this->db->getFieldNames($table);
            if (is_array($columns) && in_array('expire_date', $columns, true)) {
                $builder->select('receivings_items.expire_date');
            }
        }
        $builder->join('receivings', 'receivings_items.receiving_id = receivings.receiving_id', 'inner');
        $builder->where('receivings_items.item_id', $item_id);
        if ($location_id !== null) {
            $builder->where('receivings_items.item_location', $location_id);
        }
        if (!$minimal) {
            $table = $this->db->prefixTable('receivings_items');
            $columns = $this->db->getFieldNames($table);
            if (is_array($columns) && in_array('expire_date', $columns, true)) {
                $builder->orderBy('receivings_items.expire_date IS NULL', 'asc', false);
                $builder->orderBy('receivings_items.expire_date', 'asc');
            }
        }
        $builder->orderBy('receivings.receiving_time', 'desc');
        $builder->limit(100);

        return $builder->get()->getResultArray();
    }

    /**
     * @param int $receiving_id
     * @return object
     */
    public function get_supplier(int $receiving_id): object
    {
        $builder = $this->db->table('receivings');
        $builder->where('receiving_id', $receiving_id);

        $supplier = model(Supplier::class);
        return $supplier->get_info($builder->get()->getRow()->supplier_id);
    }

    /**
     * Get total amount for a receiving (sum of line items with discount applied).
     */
    public function get_receiving_total(int $receiving_id): float
    {
        $db_prefix = $this->db->getPrefix();
        $decimals = totals_decimals();
        $total_expr = 'CASE WHEN `' . $db_prefix . 'receivings_items`.`discount_type` = ' . PERCENT
            . ' THEN `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity`'
            . ' - ROUND(`item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` * `discount` / 100, ' . $decimals . ')'
            . ' ELSE `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` - `discount` END';

        $builder = $this->db->table('receivings_items');
        $builder->select('ROUND(SUM(' . $total_expr . '), ' . $decimals . ') AS total');
        $builder->where('receiving_id', $receiving_id);
        $row = $builder->get()->getRow();

        return $row && $row->total !== null ? (float) $row->total : 0.0;
    }

    /**
     * @return array
     */
    public function get_payment_options(): array
    {
        return get_payment_options();
    }

    /**
     * Create a temp table that allows us to do easy report/receiving queries
     */
    public function create_temp_table(array $inputs): void
    {
        $config = config(OSPOS::class)->settings;
        $db_prefix = $this->db->getPrefix();

        if (empty($inputs['receiving_id'])) {
            $where = empty($config['date_or_time_format'])
                ? 'DATE(`receiving_time`) BETWEEN ' . $this->db->escape($inputs['start_date']) . ' AND ' . $this->db->escape($inputs['end_date'])
                : 'receiving_time BETWEEN ' . $this->db->escape(rawurldecode($inputs['start_date'])) . ' AND ' . $this->db->escape(rawurldecode($inputs['end_date']));
        } else {
            $where = 'receivings_items.receiving_id = ' . $this->db->escape($inputs['receiving_id']);
        }

        $builder = $this->db->table('receivings_items');
        $builder->select([
            'MAX(DATE(`receiving_time`)) AS receiving_date',
            'MAX(`receiving_time`) AS receiving_time',
            'receivings_items.receiving_id AS receiving_id',
            'MAX(`comment`) AS comment',
            'MAX(`item_location`) AS item_location',
            'MAX(`reference`) AS reference',
            'MAX(`payment_type`) AS payment_type',
            'MAX(`employee_id`) AS employee_id',
            'items.item_id AS item_id',
            'MAX(`' . $db_prefix . 'receivings`.`supplier_id`) AS supplier_id',
            'MAX(`quantity_purchased`) AS quantity_purchased',
            ($this->db->fieldExists('carton_qty', $db_prefix . 'receivings_items')
                ? 'MAX(COALESCE(`' . $db_prefix . 'receivings_items`.`carton_qty`, 0)) AS carton_qty, MAX(COALESCE(`' . $db_prefix . 'receivings_items`.`pieces_qty`, 0)) AS pieces_qty'
                : '0 AS carton_qty, 0 AS pieces_qty'),
            'MAX(`' . $db_prefix . 'receivings_items`.`receiving_quantity`) AS item_receiving_quantity',
            'MAX(`item_cost_price`) AS item_cost_price',
            'MAX(`item_unit_price`) AS item_unit_price',
            'MAX(`discount`) AS discount',
            'MAX(`discount_type`) AS discount_type',
            'receivings_items.line AS line',
            'MAX(`serialnumber`) AS serialnumber',
            'MAX(`' . $db_prefix . 'receivings_items`.`description`) AS description',
            ($this->db->fieldExists('expire_date', $db_prefix . 'receivings_items')
                ? 'MAX(`' . $db_prefix . 'receivings_items`.`expire_date`) AS expire_date'
                : 'NULL AS expire_date'),
            'MAX(CASE WHEN `' . $db_prefix . 'receivings_items`.`discount_type` = ' . PERCENT . ' THEN `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` - `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` * `discount` / 100 ELSE `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` - `discount` END) AS subtotal',
            'MAX(CASE WHEN `' . $db_prefix . 'receivings_items`.`discount_type` = ' . PERCENT . ' THEN `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` - `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` * `discount` / 100 ELSE `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` - `discount` END) AS total',
            'MAX((CASE WHEN `' . $db_prefix . 'receivings_items`.`discount_type` = ' . PERCENT . ' THEN `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` - `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` * `discount` / 100 ELSE `item_unit_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` - discount END) - (`item_cost_price` * `quantity_purchased`)) AS profit',
            'MAX(`item_cost_price` * `quantity_purchased` * `' . $db_prefix . 'receivings_items`.`receiving_quantity` ) AS cost'
        ]);
        $builder->join('receivings', 'receivings_items.receiving_id = receivings.receiving_id', 'inner');
        $builder->join('items', 'receivings_items.item_id = items.item_id', 'inner');
        $builder->where($where);
        $builder->groupBy(['receivings_items.receiving_id', 'items.item_id', 'receivings_items.line']);
        $selectQuery = $builder->getCompiledSelect();

        // QueryBuilder does not support creating temporary tables.
        $this->db->query('DROP TEMPORARY TABLE IF EXISTS ' . $this->db->prefixTable('receivings_items_temp'));
        $sql = 'CREATE TEMPORARY TABLE ' . $this->db->prefixTable('receivings_items_temp') .
            ' (INDEX(receiving_date), INDEX(receiving_time), INDEX(receiving_id)) AS (' . $selectQuery . ')';

        $this->db->query($sql);
    }

    /**
     * Get number of rows for the receivings manage view
     */
    public function get_found_rows(string $search, array $filters): int
    {
        return $this->search($search, $filters, 0, 0, 'receivings.receiving_time', 'desc', true);
    }

    /**
     * Search receivings for the manage (Daily Receivings) view
     */
    public function search(string $search, array $filters, ?int $rows = 0, ?int $limit_from = 0, ?string $sort = 'receivings.receiving_time', ?string $order = 'desc', ?bool $count_only = false)
    {
        $rows = $rows ?? 0;
        $limit_from = $limit_from ?? 0;
        $sort = $sort ?? 'receivings.receiving_time';
        $order = $order ?? 'desc';
        $count_only = $count_only ?? false;

        $config = config(OSPOS::class)->settings;
        $db_prefix = $this->db->getPrefix();
        $decimals = totals_decimals();

        // Match get_receiving_total(): subtotal = price * qty_purchased * receiving_quantity; apply % or fixed discount
        $total_expr = 'CASE WHEN `receivings_items`.`discount_type` = ' . PERCENT
            . ' THEN `receivings_items`.`item_unit_price` * `receivings_items`.`quantity_purchased` * `receivings_items`.`receiving_quantity`'
            . ' - ROUND(`receivings_items`.`item_unit_price` * `receivings_items`.`quantity_purchased` * `receivings_items`.`receiving_quantity` * `receivings_items`.`discount` / 100, ' . $decimals . ')'
            . ' ELSE `receivings_items`.`item_unit_price` * `receivings_items`.`quantity_purchased` * `receivings_items`.`receiving_quantity` - `receivings_items`.`discount` END';

        $builder = $this->db->table('receivings AS receivings');
        $builder->select('receivings.receiving_id AS receiving_id,
            receivings.receiving_time AS receiving_time,
            COALESCE(suppliers.company_name, \'\') AS supplier_name,
            receivings.reference AS reference,
            receivings.payment_type AS payment_type,
            ROUND(SUM(' . $total_expr . '), ' . $decimals . ') AS total');

        $builder->join('receivings_items AS receivings_items', 'receivings.receiving_id = receivings_items.receiving_id', 'inner');
        $builder->join('suppliers AS suppliers', 'receivings.supplier_id = suppliers.person_id', 'left');

        $where = empty($config['date_or_time_format'])
            ? 'DATE(receivings.receiving_time) BETWEEN ' . $this->db->escape($filters['start_date']) . ' AND ' . $this->db->escape($filters['end_date'])
            : 'receivings.receiving_time BETWEEN ' . $this->db->escape(rawurldecode($filters['start_date'])) . ' AND ' . $this->db->escape(rawurldecode($filters['end_date']));
        $builder->where($where);

        if (!empty($search)) {
            $pieces = explode(' ', trim($search));
            if (count($pieces) === 2 && preg_match('/(RECV|KIT)/', $pieces[0])) {
                $builder->where('receivings.receiving_id', (int) $pieces[1]);
            } elseif ($this->get_receiving_by_reference($search)->getNumRows() > 0) {
                $builder->where('receivings.reference', $search);
            } else {
                $builder->groupStart();
                $builder->like('receivings.reference', $search);
                $builder->orWhere('receivings.receiving_id', (int) $search);
                $builder->groupEnd();
            }
        }

        $builder->groupBy('receivings.receiving_id');

        if ($count_only) {
            $countBuilder = clone $builder;
            $sub = $countBuilder->getCompiledSelect();

            return $this->db->query('SELECT COUNT(*) as cnt FROM (' . $sub . ') AS t')->getRow()->cnt;
        }

        $sort_map = [
            'receiving_id'   => 'receivings.receiving_id',
            'receiving_time' => 'receivings.receiving_time',
            'supplier_name'  => 'suppliers.company_name',
            'total'          => 'total',
            'payment_type'   => 'receivings.payment_type',
            'reference'      => 'receivings.reference'
        ];
        $sort_column = $sort_map[$sort] ?? 'receivings.receiving_id';
        $order = strtoupper($order) === 'ASC' ? 'ASC' : 'DESC';
        $builder->orderBy($sort_column, $order);

        if ($rows > 0) {
            $builder->limit($rows, $limit_from);
        }

        return $builder->get();
    }
}
