<?php

namespace App\Models\Reports;

use App\Models\Sale;
use CodeIgniter\Database\RawSql;

/**
 *
 *
 * @property sale sale
 *
 */
class Detailed_sales extends Report
{
    /**
     * @param array $inputs
     * @return void
     */
    public function create(array $inputs): void
    {
        // Create our temp tables to work with the data in our report
        $sale = model(Sale::class);
        $sale->create_temp_table($inputs);
    }

    /**
     * @return array
     */
    public function getDataColumns(): array
    {
        return [    // TODO: Duplicated code
            'summary' => [
                ['id'            => lang('Reports.sale_id')],
                ['type_code'     => lang('Reports.code_type')],
                ['sale_time'     => lang('Reports.date'), 'sortable' => false],
                ['carton_qty'    => lang('Sales.carton_qty')],
                ['pieces_qty'   => lang('Sales.pieces_qty')],
                ['employee_name' => lang('Reports.sold_by')],
                ['customer_name' => lang('Reports.sold_to')],
                ['subtotal'      => lang('Reports.subtotal'), 'sorter' => 'number_sorter'],
                ['tax'           => lang('Reports.tax'), 'sorter' => 'number_sorter'],
                ['total'         => lang('Reports.total'), 'sorter' => 'number_sorter'],
                ['cost'          => lang('Reports.cost'), 'sorter' => 'number_sorter'],
                ['profit'        => lang('Reports.profit'), 'sorter' => 'number_sorter'],
                ['payment_type'  => lang('Reports.payment_type'), 'sortable' => false],
                ['comment'       => lang('Reports.comments')]
            ],
            'details' => [
                lang('Reports.name'),
                lang('Reports.category'),
                lang('Reports.item_number'),
                lang('Reports.description'),
                lang('Sales.carton_qty'),
                lang('Sales.pieces_qty'),
                lang('Reports.subtotal'),
                lang('Reports.tax'),
                lang('Reports.total'),
                lang('Reports.cost'),
                lang('Reports.profit'),
                lang('Reports.discount')
            ],
            'details_rewards' => [
                lang('Reports.used'),
                lang('Reports.earned')
            ]
        ];
    }

    /**
     * @param int $sale_id
     * @return array
     */
    public function getDataBySaleId(int $sale_id): array
    {
        $cartonPieces = $this->db->fieldExists('carton_qty', $this->db->getPrefix() . 'sales_items')
            ? 'SUM(COALESCE(carton_qty, 0)) AS carton_qty, SUM(COALESCE(pieces_qty, 0)) AS pieces_qty, '
            : '0 AS carton_qty, 0 AS pieces_qty, ';
        $builder = $this->db->table('sales_items_temp');
        $builder->select('sale_id,
            MAX(sale_time) as sale_time,
            SUM(quantity_purchased) AS items_purchased,
            ' . $cartonPieces . '
            MAX(employee_name) AS employee_name,
            MAX(customer_name) AS customer_name,
            SUM(subtotal) AS subtotal,
            SUM(tax) AS tax,
            SUM(total) AS total,
            SUM(cost) AS cost,
            SUM(profit) AS profit,
            MAX(payment_type) AS payment_type,
            MAX(sale_status) AS sale_status,
            comment');
        $builder->where('sale_id', $sale_id);

        return $builder->get()->getRowArray();
    }

    /**
     * Get item details for a single sale by querying sales_items + items directly.
     * Bypasses sales_items_temp to avoid column/structure issues.
     */
    private function getDetailsForSale(int $sale_id, array $inputs): array
    {
        $decimals = totals_decimals();
        $prefix = $this->db->getPrefix() ?: '';
        $si = 'si';
        $it = 'it';
        $sit = 'sit';

        $sale_price = "CASE WHEN {$si}.discount_type = " . PERCENT
            . " THEN {$si}.quantity_purchased * {$si}.item_unit_price - ROUND({$si}.quantity_purchased * {$si}.item_unit_price * {$si}.discount / 100, {$decimals}) "
            . "ELSE {$si}.quantity_purchased * ({$si}.item_unit_price - {$si}.discount) END";
        $cost_expr = "({$si}.item_cost_price * {$si}.quantity_purchased)";
        $carton_cols = $this->db->fieldExists('carton_qty', $prefix . 'sales_items')
            ? "COALESCE({$si}.carton_qty, 0) AS carton_qty, COALESCE({$si}.pieces_qty, 0) AS pieces_qty"
            : "0 AS carton_qty, 0 AS pieces_qty";

        $hasDefIds = !empty($inputs['definition_ids']);
        $agg = $hasDefIds ? 'MAX' : '';
        $select = ($agg ? "{$agg}({$it}.name) AS name, {$agg}({$it}.category) AS category, {$agg}({$si}.quantity_purchased) AS quantity_purchased, {$agg}({$si}.item_location) AS item_location, {$agg}({$it}.item_number) AS item_number, {$agg}({$si}.description) AS description" : "{$it}.name, {$it}.category, {$si}.quantity_purchased, {$si}.item_location, {$it}.item_number, {$si}.description");
        $taxCol = $agg ? "IFNULL({$agg}({$sit}.tax), 0)" : "IFNULL({$sit}.tax, 0)";
        $salesTaxCol = $agg ? "IFNULL({$agg}({$sit}.sales_tax), 0)" : "IFNULL({$sit}.sales_tax, 0)";
        $select .= ",
            ROUND({$sale_price}, {$decimals}) AS subtotal,
            {$taxCol} AS tax,
            ROUND({$sale_price}, {$decimals}) + {$salesTaxCol} AS total,
            {$cost_expr} AS cost,
            ROUND({$sale_price}, {$decimals}) - {$cost_expr} AS profit,
            {$si}.discount, {$si}.discount_type, {$si}.item_id, {$carton_cols}";

        $builder = $this->db->table('sales_items AS ' . $si);
        $builder->select(new RawSql($select));
        $builder->join('items AS ' . $it, "{$si}.item_id = {$it}.item_id", 'inner');
        $builder->join('sales_items_taxes_temp AS ' . $sit, "{$si}.sale_id = {$sit}.sale_id AND {$si}.item_id = {$sit}.item_id AND {$si}.line = {$sit}.line", 'left');
        $builder->where("{$si}.sale_id", $sale_id);

        if ($inputs['location_id'] !== 'all') {
            $builder->where("{$si}.item_location", $inputs['location_id']);
        }

        if ($this->db->fieldExists('item_unit', $prefix . 'items')) {
            $builder->select(new RawSql("{$it}.item_unit, {$it}.qty_per_unit, {$it}.qty_per_carton, {$it}.qty_per_box, {$it}.qty_per_bag, {$it}.qty_per_pack, {$it}.pack_name"));
        }

        if (!empty($inputs['definition_ids'])) {
            $def_ids = implode(',', array_map('intval', $inputs['definition_ids']));
            $builder->join('attribute_links', "attribute_links.item_id = {$si}.item_id AND attribute_links.sale_id = {$si}.sale_id AND definition_id IN ({$def_ids})", 'left');
            $builder->join('attribute_values', 'attribute_values.attribute_id = attribute_links.attribute_id', 'left');
            $fmt = $this->db->escape(dateformat_mysql());
            $builder->select(new RawSql("GROUP_CONCAT(DISTINCT CONCAT_WS('_', attribute_links.definition_id, attribute_values.attribute_value) ORDER BY attribute_links.definition_id SEPARATOR '|') AS attribute_values"));
            $builder->select(new RawSql("GROUP_CONCAT(DISTINCT CONCAT_WS('_', attribute_links.definition_id, DATE_FORMAT(attribute_values.attribute_date, $fmt)) SEPARATOR '|') AS attribute_dtvalues"));
            $builder->select(new RawSql("GROUP_CONCAT(DISTINCT CONCAT_WS('_', attribute_links.definition_id, attribute_values.attribute_decimal) SEPARATOR '|') AS attribute_dvalues"));
            $builder->groupBy("{$si}.sale_id, {$si}.item_id, {$si}.line");
        }

        return $builder->get()->getResultArray();
    }

    /**
     * @param array $inputs
     * @return array
     */
    public function getData(array $inputs): array
    {
        $builder = $this->db->table('sales_items_temp');
        $builder->select('sale_id,
            MAX(CASE
            WHEN sale_type = ' . SALE_TYPE_POS . ' && sale_status = ' . COMPLETED . ' THEN \'' . lang('Reports.code_pos') . '\'
            WHEN sale_type = ' . SALE_TYPE_INVOICE . ' && sale_status = ' . COMPLETED . ' THEN \'' . lang('Reports.code_invoice') . '\'
            WHEN sale_type = ' . SALE_TYPE_WORK_ORDER . ' && sale_status = ' . SUSPENDED . ' THEN \'' . lang('Reports.code_work_order') . '\'
            WHEN sale_type = ' . SALE_TYPE_QUOTE . ' && sale_status = ' . SUSPENDED . ' THEN \'' . lang('Reports.code_quote') . '\'
            WHEN sale_type = ' . SALE_TYPE_RETURN . ' && sale_status = ' . COMPLETED . ' THEN \'' . lang('Reports.code_return') . '\'
            WHEN sale_status = ' . CANCELED . ' THEN \'' . lang('Reports.code_canceled') . '\'
            ELSE \'\'
            END) AS type_code,
            MAX(sale_status) as sale_status,
            MAX(sale_time) AS sale_time,
            ' . ($this->db->fieldExists('carton_qty', $this->db->getPrefix() . 'sales_items')
                ? 'SUM(COALESCE(carton_qty, 0)) AS carton_qty, SUM(COALESCE(pieces_qty, 0)) AS pieces_qty, '
                : '0 AS carton_qty, 0 AS pieces_qty, ') . '
            MAX(employee_name) AS employee_name,
            MAX(customer_name) AS customer_name,
            SUM(subtotal) AS subtotal,
            SUM(tax) AS tax,
            SUM(total) AS total,
            SUM(cost) AS cost,
            SUM(profit) AS profit,
            MAX(payment_type) AS payment_type,
            MAX(comment) AS comment');

        if ($inputs['location_id'] != 'all') {    // TODO: Duplicated code
            $builder->where('item_location', $inputs['location_id']);
        }

        switch ($inputs['sale_type']) {
            case 'complete':
                $builder->where('sale_status', COMPLETED);
                $builder->groupStart();
                $builder->where('sale_type', SALE_TYPE_POS);
                $builder->orWhere('sale_type', SALE_TYPE_INVOICE);
                $builder->orWhere('sale_type', SALE_TYPE_RETURN);
                $builder->groupEnd();
                break;

            case 'sales':
                $builder->where('sale_status', COMPLETED);
                $builder->groupStart();
                $builder->where('sale_type', SALE_TYPE_POS);
                $builder->orWhere('sale_type', SALE_TYPE_INVOICE);
                $builder->groupEnd();
                break;

            case 'quotes':
                $builder->where('sale_status', SUSPENDED);
                $builder->where('sale_type', SALE_TYPE_QUOTE);
                break;

            case 'work_orders':
                $builder->where('sale_status', SUSPENDED);
                $builder->where('sale_type', SALE_TYPE_WORK_ORDER);
                break;

            case 'canceled':
                $builder->where('sale_status', CANCELED);
                break;

            case 'returns':
                $builder->where('sale_status', COMPLETED);
                $builder->where('sale_type', SALE_TYPE_RETURN);
                break;
        }

        $builder->groupBy('sale_id');
        $builder->orderBy('MAX(sale_time)');

        $data = [];
        $data['summary'] = $builder->get()->getResultArray();
        $data['details'] = [];
        $data['rewards'] = [];

        foreach ($data['summary'] as $key => $value) {
            $data['details'][$key] = $this->getDetailsForSale($value['sale_id'], $inputs);
            $builder = $this->db->table('sales_reward_points');
            $builder->where('sale_id', $value['sale_id']);
            $data['rewards'][$key] = $builder->get()->getResultArray();
        }

        return $data;
    }

    /**
     * @param array $inputs
     * @return array
     */
    public function getSummaryData(array $inputs): array
    {
        $builder = $this->db->table('sales_items_temp');
        $builder->select('SUM(subtotal) AS subtotal, SUM(tax) AS tax, SUM(total) AS total, SUM(cost) AS cost, SUM(profit) AS profit');

        if ($inputs['location_id'] != 'all') {    // TODO: Duplicated code
            $builder->where('item_location', $inputs['location_id']);
        }

        switch ($inputs['sale_type']) {
            case 'complete':
                $builder->where('sale_status', COMPLETED);
                $builder->groupStart();
                $builder->where('sale_type', SALE_TYPE_POS);
                $builder->orWhere('sale_type', SALE_TYPE_INVOICE);
                $builder->orWhere('sale_type', SALE_TYPE_RETURN);
                $builder->groupEnd();
                break;

            case 'sales':
                $builder->where('sale_status', COMPLETED);
                $builder->groupStart();
                $builder->where('sale_type', SALE_TYPE_POS);
                $builder->orWhere('sale_type', SALE_TYPE_INVOICE);
                $builder->groupEnd();
                break;

            case 'quotes':
                $builder->where('sale_status', SUSPENDED);
                $builder->where('sale_type', SALE_TYPE_QUOTE);
                break;

            case 'work_orders':
                $builder->where('sale_status', SUSPENDED);
                $builder->where('sale_type', SALE_TYPE_WORK_ORDER);
                break;

            case 'canceled':
                $builder->where('sale_status', CANCELED);
                break;

            case 'returns':
                $builder->where('sale_status', COMPLETED);
                $builder->where('sale_type', SALE_TYPE_RETURN);
                break;
        }

        return $builder->get()->getRowArray();
    }
}
