<?php

namespace App\Models\Reports;

use App\Models\Item;

/**
 * Report for items that are expired or nearing expiration (based on items.expire_date).
 */
class Inventory_expired extends Report
{
    /**
     * @return array[]
     */
    public function getDataColumns(): array
    {
        return [
            ['item_name'     => lang('Reports.item_name')],
            ['item_number'   => lang('Reports.item_number')],
            ['category'      => lang('Reports.category')],
            ['carton_qty'    => lang('Sales.carton_qty')],
            ['pieces_qty'    => lang('Sales.pieces_qty')],
            ['expire_date'   => lang('Items.expire_date')],
            ['location_name' => lang('Reports.stock_location')]
        ];
    }

    /**
     * @param array $inputs ['location_id' => string, 'days_ahead' => int]
     * @return array
     */
    public function getData(array $inputs): array
    {
        $itemsTable = $this->db->getPrefix() . 'items';
        if (!$this->db->fieldExists('expire_date', $itemsTable)) {
            return [];
        }

        $item = model(Item::class);
        $days_ahead = (int) ($inputs['days_ahead'] ?? 30);
        $cutoff_date = date('Y-m-d', strtotime("+{$days_ahead} days"));

        $builder = $this->db->table('items AS items');
        $itemCols = $this->db->fieldExists('item_unit', $itemsTable)
            ? ', items.item_unit, items.qty_per_unit, items.qty_per_carton, items.qty_per_box, items.qty_per_bag, items.qty_per_pack, items.pack_name'
            : '';
        $builder->select(
            $item->get_item_name('name') . ',
            items.item_number,
            items.category,
            item_quantities.quantity,
            items.expire_date,
            stock_locations.location_name' . $itemCols
        );
        $builder->join('item_quantities AS item_quantities', 'items.item_id = item_quantities.item_id');
        $builder->join('stock_locations', 'item_quantities.location_id = stock_locations.location_id');
        $builder->where('items.deleted', 0);
        $builder->where('items.stock_type', 0);
        $builder->where('items.expire_date IS NOT NULL');
        $builder->where('items.expire_date <=', $cutoff_date);
        $builder->where('stock_locations.deleted', 0);

        if (!empty($inputs['location_id']) && $inputs['location_id'] !== 'all') {
            $builder->where('stock_locations.location_id', $inputs['location_id']);
        }

        $builder->orderBy('items.expire_date', 'asc');
        $builder->orderBy('items.name', 'asc');

        return $builder->get()->getResultArray();
    }

    /**
     * @param array $inputs
     * @return array
     */
    public function getSummaryData(array $inputs): array
    {
        $data = $this->getData($inputs);
        $summary = ['total_quantity' => 0];
        foreach ($data as $row) {
            $summary['total_quantity'] += (float) ($row['quantity'] ?? 0);
        }
        return $summary;
    }

    /**
     * @return array [days => label]
     */
    public function getDaysAheadOptions(): array
    {
        return [
            0   => lang('Reports.expired_only'),
            7   => lang('Reports.days_7'),
            14  => lang('Reports.days_14'),
            30  => lang('Reports.days_30'),
            60  => lang('Reports.days_60'),
            90  => lang('Reports.days_90')
        ];
    }
}
