<?php

namespace App\Models\Reports\Traits;

use CodeIgniter\Database\BaseBuilder;

/**
 * Trait to apply sale type filter to a query builder.
 * Use $tablePrefix for the table/alias prefix when columns are qualified (e.g. 'sales.' or '').
 */
trait SaleTypeFilter
{
    /**
     * Apply location filter to builder when location_id is not 'all'.
     *
     * @param BaseBuilder $builder   Query builder
     * @param string      $locationId Value from inputs
     * @param string      $column    Location column (e.g. 'sales_items.item_location' or 'item_location')
     */
    protected function applyLocationFilter(BaseBuilder $builder, string $locationId, string $column = 'sales_items.item_location'): void
    {
        if ($locationId !== 'all' && $locationId !== '') {
            $builder->where($column, (int) $locationId);
        }
    }

    /**
     * Apply sale type filter to builder.
     *
     * @param BaseBuilder $builder Query builder
     * @param string      $saleType One of: complete, sales, quotes, work_orders, canceled, returns
     * @param string      $tablePrefix Prefix for status/type columns (e.g. 'sales.' or '')
     */
    protected function applySaleTypeFilter(BaseBuilder $builder, string $saleType, string $tablePrefix = ''): void
    {
        $status = $tablePrefix . 'sale_status';
        $type   = $tablePrefix . 'sale_type';

        switch ($saleType) {
            case 'complete':
                $builder->where($status, COMPLETED);
                $builder->groupStart();
                $builder->where($type, SALE_TYPE_POS);
                $builder->orWhere($type, SALE_TYPE_INVOICE);
                $builder->orWhere($type, SALE_TYPE_RETURN);
                $builder->groupEnd();
                break;

            case 'sales':
                $builder->where($status, COMPLETED);
                $builder->groupStart();
                $builder->where($type, SALE_TYPE_POS);
                $builder->orWhere($type, SALE_TYPE_INVOICE);
                $builder->groupEnd();
                break;

            case 'quotes':
                $builder->where($status, SUSPENDED);
                $builder->where($type, SALE_TYPE_QUOTE);
                break;

            case 'work_orders':
                $builder->where($status, SUSPENDED);
                $builder->where($type, SALE_TYPE_WORK_ORDER);
                break;

            case 'canceled':
                $builder->where($status, CANCELED);
                break;

            case 'returns':
                $builder->where($status, COMPLETED);
                $builder->where($type, SALE_TYPE_RETURN);
                break;
        }
    }
}
