<?php

namespace app\Libraries;

use App\Models\Attribute;
use App\Models\Item;
use App\Models\Item_kit_items;
use App\Models\Item_quantity;
use App\Models\Receiving;
use App\Models\Stock_location;

use CodeIgniter\Session\Session;
use Config\OSPOS;

/**
 * Receiving library
 *
 * Library with utilities to manage receivings
 */
class Receiving_lib
{
    private Attribute $attribute;
    private Item $item;
    private Item_kit_items $item_kit_items;
    private Item_quantity $item_quantity;
    private Receiving $receiving;
    private Stock_location $stock_location;
    private Session $session;

    public function __construct()
    {
        $this->attribute = model(Attribute::class);
        $this->item = model(Item::class);
        $this->item_kit_items = model(Item_kit_items::class);
        $this->item_quantity = model(Item_quantity::class);
        $this->receiving = model(Receiving::class);
        $this->stock_location = model(Stock_location::class);

        $this->session = session();
    }

    /**
     * @return array
     */
    public function get_cart(): array
    {
        if (!$this->session->get('recv_cart')) {
            $this->set_cart([]);
        }

        return $this->session->get('recv_cart');
    }

    /**
     * @param array $cart_data
     * @return void
     */
    public function set_cart(array $cart_data): void
    {
        $this->session->set('recv_cart', $cart_data);
    }

    /**
     * @return void
     */
    public function empty_cart(): void
    {
        $this->session->remove('recv_cart');
    }

    /**
     * @return int
     */
    public function get_supplier(): int
    {
        if (!$this->session->get('recv_supplier')) {
            $this->set_supplier(-1);    // TODO: Replace -1 with a constant.
        }

        return $this->session->get('recv_supplier');
    }

    /**
     * @param int $supplier_id
     * @return void
     */
    public function set_supplier(int $supplier_id): void
    {
        $this->session->set('recv_supplier', $supplier_id);
    }

    /**
     * @return void
     */
    public function remove_supplier(): void
    {
        $this->session->remove('recv_supplier');
    }

    /**
     * @return string
     */
    public function get_mode(): string
    {
        if (!$this->session->get('recv_mode')) {
            $this->set_mode('receive');
        }

        return $this->session->get('recv_mode');
    }

    /**
     * @param string $mode
     * @return void
     */
    public function set_mode(string $mode): void
    {
        $this->session->set('recv_mode', $mode);
    }

    /**
     * @return void
     */
    public function clear_mode(): void    // TODO: This function verb is inconsistent from the others.  Consider refactoring to remove_mode()
    {
        $this->session->remove('recv_mode');
    }

    /**
     * @return int
     */
    public function get_stock_source(): int
    {
        if (!$this->session->get('recv_stock_source')) {
            $this->set_stock_source($this->stock_location->get_default_location_id('receivings'));
        }

        return $this->session->get('recv_stock_source');
    }

    /**
     * @return string
     */
    public function get_comment(): string
    {
        $comment = $this->session->get('recv_comment');

        return empty($comment) ? '' : $comment;
    }

    /**
     * @param string $comment
     * @return void
     */
    public function set_comment(string $comment): void
    {
        $this->session->set('recv_comment', $comment);
    }

    /**
     * @return void
     */
    public function clear_comment(): void    // TODO: This function verb is inconsistent from the others.  Consider refactoring to remove_comment()
    {
        $this->session->remove('recv_comment');
    }

    /**
     * @return string
     */
    public function get_reference(): string
    {
        return $this->session->get('recv_reference') ?? '';
    }

    /**
     * @param string $reference
     * @return void
     */
    public function set_reference(string $reference): void
    {
        $this->session->set('recv_reference', $reference);
    }

    /**
     * @return void
     */
    public function clear_reference(): void    // TODO: This function verb is inconsistent from the others.  Consider refactoring to remove_reference()
    {
        $this->session->remove('recv_reference');
    }

    /**
     * Set receiving ID for edit mode. Use -1 or clear when creating new.
     *
     * @param int $receiving_id
     * @return void
     */
    public function set_receiving_id(int $receiving_id): void
    {
        $this->session->set('recv_editing_id', $receiving_id);
    }

    /**
     * Get receiving ID when editing. Returns -1 when creating new.
     *
     * @return int
     */
    public function get_receiving_id(): int
    {
        $id = $this->session->get('recv_editing_id');

        return $id !== null ? (int) $id : -1;
    }

    /**
     * Clear the editing receiving ID.
     *
     * @return void
     */
    public function clear_receiving_id(): void
    {
        $this->session->remove('recv_editing_id');
    }

    /**
     * @param string|null $payment_type
     * @return void
     */
    public function set_payment_type(?string $payment_type): void
    {
        $this->session->set('recv_payment_type', $payment_type ?? '');
    }

    /**
     * @return string
     */
    public function get_payment_type(): string
    {
        return $this->session->get('recv_payment_type') ?? '';
    }

    /**
     * @return void
     */
    public function clear_payment_type(): void
    {
        $this->session->remove('recv_payment_type');
    }

    /**
     * @return bool
     */
    public function is_print_after_sale(): bool
    {
        return $this->session->get('recv_print_after_sale') == 'true'
            || $this->session->get('recv_print_after_sale') == '1';
    }

    /**
     * @param bool $print_after_sale
     * @return void
     */
    public function set_print_after_sale(bool $print_after_sale): void
    {
        $this->session->set('recv_print_after_sale', $print_after_sale);
    }

    /**
     * @param int $stock_source
     * @return void
     */
    public function set_stock_source(int $stock_source): void
    {
        $this->session->set('recv_stock_source', $stock_source);
    }

    /**
     * @return void
     */
    public function clear_stock_source(): void
    {
        $this->session->remove('recv_stock_source');
    }

    /**
     * @return string
     */
    public function get_stock_destination(): string
    {
        if (!$this->session->get('recv_stock_destination')) {
            $this->set_stock_destination($this->stock_location->get_default_location_id('receivings'));
        }

        return $this->session->get('recv_stock_destination');
    }

    /**
     * @param string $stock_destination
     * @return void
     */
    public function set_stock_destination(?string $stock_destination): void
    {
        $this->session->set('recv_stock_destination', $stock_destination);
    }

    /**
     * @return void
     */
    public function clear_stock_destination(): void
    {
        $this->session->remove('recv_stock_destination');
    }
    // TODO: This array signature needs to be reworked.  It's way too long. Perhaps an object needs to be passed rather than these?

    /**
     * @param string $itemId
     * @param int $quantity
     * @param int|null $itemLocation
     * @param float $discount
     * @param int $discountType
     * @param float|null $price
     * @param string|null $description
     * @param string|null $serialNumber
     * @param float|null $receivingQuantity
     * @param int|null $receivingId
     * @param bool $includeDeleted
     * @param string|null $expireDate Batch expire date; same item with different expire = separate line
     * @return bool
     */
    public function add_item(string $itemId, int $quantity = 1, ?int $itemLocation = null, float $discount = 0, int $discountType = 0, ?float $price = null, ?string $description = null, ?string $serialNumber = null, ?float $receivingQuantity = null, ?int $receivingId = null, bool $includeDeleted = false, ?string $expireDate = null): bool
    {
        $config = config(OSPOS::class)->settings;
        $itemInfo = $this->item->get_info_by_id_or_number($itemId, $includeDeleted);

        if (empty($itemInfo)) {
            return false;
        }

        $itemId = $itemInfo->item_id;
        $items = $this->get_cart();

        $maxKey = 0;
        $itemAlreadyInSale = false;
        $updateKey = 0;

        $expireNormalized = !empty($expireDate) ? $expireDate : null;
        if ($expireNormalized === null && !empty($itemInfo->expire_date ?? null)) {
            $expireNormalized = $itemInfo->expire_date;
        }

        foreach ($items as $item) {
            if ($maxKey <= $item['line']) {
                $maxKey = $item['line'];
            }

            $itemExpire = $item['expire_date'] ?? null;
            $itemExpire = !empty($itemExpire) ? $itemExpire : null;
            $expireMatch = ($itemExpire === $expireNormalized)
                || ($itemExpire === null && $expireNormalized === null);

            if ($item['item_id'] == $itemId && $item['item_location'] == $itemLocation && $expireMatch) {
                $itemAlreadyInSale = true;
                $updateKey = $item['line'];
            }
        }

        $insertKey = $maxKey + 1;
        $itemInfo = $this->item->get_info((int) $itemId);

        $price = $price != null ? $price : 0;

        if ($config['multi_pack_enabled']) {
            $itemInfo->name .= NAME_SEPARATOR . $itemInfo->pack_name;
        }

        $availableUnits = $this->item->get_available_units($itemInfo);
        $receivingQuantityChoices = [];
        foreach ($availableUnits as $unit => $info) {
            $qty = (float) $info[0];
            $label = $info[1];
            $key = (string) round($qty, quantity_decimals());
            $receivingQuantityChoices[$key] = $qty > 1 ? $label . ' (x' . $key . ')' : $label;
        }
        if (empty($receivingQuantityChoices)) {
            $receivingQuantityChoices = ['1' => 'Pieces'];
        }

        if (is_null($receivingQuantity)) {
            $receivingQuantity = $itemInfo->receiving_quantity;
            if ($receivingQuantity == 0) {
                $receivingQuantity = 1;
            }
        }

        $attributeLinks = $this->attribute->get_link_values((int) $itemId, 'receiving_id', $receivingId, Attribute::SHOW_IN_RECEIVINGS)->getRowObject();

        $priceStr = (string) $price;
        $discountStr = (string) $discount;
        $rq = (float) $receivingQuantity;
        $useBulk = $rq > 1;
        $qtyCarton = $useBulk ? (string) $quantity : '0';
        $qtyPiece = $useBulk ? '0' : (string) $quantity;
        $quantityPieces = $useBulk
            ? bcmul((string) $quantity, (string) $rq, quantity_decimals())
            : (string) $quantity;
        if ($useBulk) {
            $unitPriceBulk = bcmul($priceStr, (string) $rq, totals_decimals());
            $pricePerPiece = $priceStr;
        } else {
            $unitPriceBulk = $priceStr;
            $pricePerPiece = $priceStr;
        }

        $baseItem = [
            'item_id'                    => $itemId,
            'item_location'              => $itemLocation,
            'item_number'                => $itemInfo->item_number,
            'stock_name'                 => $this->stock_location->get_location_name($itemLocation),
            'line'                       => $insertKey,
            'name'                       => $itemInfo->name,
            'pic_filename'               => $itemInfo->pic_filename ?? null,
            'description'                => $description != null ? $description : $itemInfo->description,
            'serialnumber'               => $serialNumber != null ? $serialNumber : '',
            'attribute_values'           => $attributeLinks->attribute_values,
            'attribute_dtvalues'         => $attributeLinks->attribute_dtvalues,
            'allow_alt_description'      => $itemInfo->allow_alt_description,
            'is_serialized'              => $itemInfo->is_serialized,
            'quantity'                   => $quantity,
            'discount'                   => $discount,
            'discount_type'              => $discountType,
            'in_stock'                   => $this->item_quantity->get_item_quantity((int) $itemId, $itemLocation)->quantity,
            'price'                      => $price,
            'receiving_quantity'         => $receivingQuantity,
            'receiving_quantity_choices' => $receivingQuantityChoices,
            'expire_date'                => $expireNormalized,
        ];
        if ($useBulk) {
            $baseItem['qty_carton'] = $qtyCarton;
            $baseItem['qty_piece'] = $qtyPiece;
            $baseItem['unit_price_bulk'] = $unitPriceBulk;
            $baseItem['price_per_piece'] = $pricePerPiece;
            $baseItem['total'] = $this->get_item_total_bulk($qtyCarton, $qtyPiece, $unitPriceBulk, $pricePerPiece, $quantityPieces, $discountStr, $discountType, true);
        } else {
            $baseItem['total'] = $this->get_item_total($quantity, $price, $discount, $discountType, $receivingQuantity);
        }
        $item = [$insertKey => $baseItem];

        if ($itemAlreadyInSale) {
            $items[$updateKey]['quantity'] += $quantity;
            if ($useBulk && isset($items[$updateKey]['qty_carton'], $items[$updateKey]['qty_piece'], $items[$updateKey]['unit_price_bulk'], $items[$updateKey]['price_per_piece'])) {
                $items[$updateKey]['qty_carton'] = bcadd($items[$updateKey]['qty_carton'], $qtyCarton, quantity_decimals());
                $qtyPiecesNew = bcadd(bcmul($items[$updateKey]['qty_carton'], (string) $items[$updateKey]['receiving_quantity'], quantity_decimals()), $items[$updateKey]['qty_piece'], quantity_decimals());
                $items[$updateKey]['total'] = $this->get_item_total_bulk($items[$updateKey]['qty_carton'], $items[$updateKey]['qty_piece'], $items[$updateKey]['unit_price_bulk'], $items[$updateKey]['price_per_piece'], $qtyPiecesNew, (string) $items[$updateKey]['discount'], $items[$updateKey]['discount_type'], true);
            } else {
                $items[$updateKey]['total'] = $this->get_item_total($items[$updateKey]['quantity'], $price, $discount, $discountType, $items[$updateKey]['receiving_quantity']);
            }
            if ($expireNormalized !== null) {
                $items[$updateKey]['expire_date'] = $expireNormalized;
            }
        } else {
            $items += $item;
        }

        $this->set_cart($items);

        return true;
    }

    /**
     * @param int|string $line
     * @param string $description
     * @param string $serialnumber
     * @param float $quantity
     * @param float $discount
     * @param int|null $discount_type
     * @param float $price
     * @param float $receiving_quantity
     * @param string|null $qty_carton For bulk items
     * @param string|null $qty_piece For bulk items
     * @param string|null $unit_price_bulk For bulk items
     * @param string|null $price_per_piece For bulk items
     * @param string|null $expire_date Batch expire date
     * @return bool
     */
    public function edit_item($line, string $description, string $serialnumber, float $quantity, float $discount, ?int $discount_type, float $price, float $receiving_quantity, ?string $qty_carton = null, ?string $qty_piece = null, ?string $unit_price_bulk = null, ?string $price_per_piece = null, ?string $expire_date = null): bool
    {
        $items = $this->get_cart();
        if (isset($items[$line])) {
            $lineRef = &$items[$line];
            $lineRef['description'] = $description;
            $lineRef['serialnumber'] = $serialnumber;
            $lineRef['quantity'] = $quantity;
            $lineRef['receiving_quantity'] = $receiving_quantity;
            $lineRef['discount'] = $discount;

            if ($discount_type !== null) {
                $lineRef['discount_type'] = $discount_type;
            }

            if ($expire_date !== null) {
                $lineRef['expire_date'] = $expire_date !== '' ? $expire_date : null;
            }

            $lineRef['price'] = $price;
            $dt = (int) ($lineRef['discount_type'] ?? 0);
            $dStr = (string) $discount;

            if ($qty_carton !== null && $qty_piece !== null && $unit_price_bulk !== null && $price_per_piece !== null && $receiving_quantity > 1) {
                $lineRef['qty_carton'] = $qty_carton;
                $lineRef['qty_piece'] = $qty_piece;
                $lineRef['unit_price_bulk'] = $unit_price_bulk;
                $lineRef['price_per_piece'] = $price_per_piece;
                $quantityPieces = bcadd(bcmul($qty_carton, (string) $receiving_quantity, quantity_decimals()), $qty_piece, quantity_decimals());
                $lineRef['total'] = $this->get_item_total_bulk($qty_carton, $qty_piece, $unit_price_bulk, $price_per_piece, $quantityPieces, $dStr, $dt, true);
            } else {
                unset($lineRef['qty_carton'], $lineRef['qty_piece'], $lineRef['unit_price_bulk'], $lineRef['price_per_piece']);
                $lineRef['total'] = $this->get_item_total($quantity, $price, $discount, $discount_type, $receiving_quantity);
            }
            $this->set_cart($items);
        }

        return false;
    }

    /**
     * @param $line int|string The item_number or item_id of the item to be removed from the receiving.
     */
    public function delete_item($line): void
    {
        $items = $this->get_cart();
        unset($items[$line]);
        $this->set_cart($items);
    }

    /**
     * @param int $receipt_receiving_id
     * @return void
     */
    public function return_entire_receiving(int $receipt_receiving_id): void
    {
        // RECV #
        $pieces = explode(' ', $receipt_receiving_id);

        if (preg_match("/(RECV|KIT)/", $pieces[0])) {    // TODO: this needs to be converted to ternary notation.
            $receiving_id = $pieces[1];
        } else {
            $receiving_id = $this->receiving->get_receiving_by_reference($receipt_receiving_id)->getRow()->receiving_id;
        }

        $this->empty_cart();
        $this->remove_supplier();
        $this->clear_comment();

        foreach ($this->receiving->get_receiving_items($receiving_id)->getResult() as $row) {
            $this->add_item($row->item_id, -$row->quantity_purchased, $row->item_location, $row->discount, $row->discount_type, $row->item_unit_price, $row->description, $row->serialnumber, $row->receiving_quantity, $receiving_id, true);
        }

        $this->set_supplier($this->receiving->get_supplier($receiving_id)->person_id);
    }

    /**
     * @param string $external_item_kit_id
     * @param int $item_location
     * @param float $discount
     * @param int $discount_type
     * @return void
     */
    public function add_item_kit(string $external_item_kit_id, int $item_location, float $discount, int $discount_type): void
    {
        // KIT #
        $pieces = explode(' ', $external_item_kit_id);
        $item_kit_id = count($pieces) > 1 ? $pieces[1] : $external_item_kit_id;

        foreach ($this->item_kit_items->get_info($item_kit_id) as $item_kit_item) {
            $this->add_item($item_kit_item['item_id'], $item_kit_item['quantity'], $item_location, $discount, $discount_type);
        }
    }

    /**
     * @param int $receiving_id
     * @return void
     */
    public function copy_entire_receiving(int $receiving_id): void
    {
        $this->empty_cart();
        $this->remove_supplier();

        foreach ($this->receiving->get_receiving_items($receiving_id)->getResult() as $row) {
            $qp = (float) ($row->quantity_purchased ?? 0);
            $rq = isset($row->receiving_quantity) ? (float) $row->receiving_quantity : 1.0;
            $expireDate = isset($row->expire_date) && $row->expire_date !== '' ? $row->expire_date : null;

            if ($rq > 1) {
                $this->add_item($row->item_id, (int) $qp, $row->item_location, $row->discount ?? 0, $row->discount_type ?? 0, $row->item_unit_price, $row->description ?? '', $row->serialnumber ?? '', $rq, $receiving_id, true, $expireDate);
            } else {
                $itemInfo = $this->item->get_info((int) $row->item_id);
                $availableUnits = $this->item->get_available_units($itemInfo);
                $itemRq = (float) ($itemInfo->receiving_quantity ?? 0);
                if ($itemRq <= 1) {
                    foreach (['carton', 'box', 'pack', 'bag'] as $unit) {
                        if (isset($availableUnits[$unit]) && ($availableUnits[$unit][0] ?? 0) > 1) {
                            $itemRq = (float) $availableUnits[$unit][0];
                            break;
                        }
                    }
                }
                if ($itemRq <= 1) {
                    $itemRq = 1.0;
                }
                if ($itemRq > 1) {
                    $qtyCarton = (int) floor($qp / $itemRq);
                    $qtyPiece = $qp - ($qtyCarton * $itemRq);
                    $this->add_item($row->item_id, $qtyCarton, $row->item_location, $row->discount ?? 0, $row->discount_type ?? 0, $row->item_unit_price, $row->description ?? '', $row->serialnumber ?? '', $itemRq, $receiving_id, true, $expireDate);
                    if ($qtyPiece != 0) {
                        $cart = $this->get_cart();
                        $line = array_key_last($cart);
                        if ($line !== null) {
                            $cart[$line]['qty_piece'] = (string) $qtyPiece;
                            $quantityPieces = bcadd(bcmul((string) $qtyCarton, (string) $itemRq, quantity_decimals()), (string) $qtyPiece, quantity_decimals());
                            $cart[$line]['total'] = $this->get_item_total_bulk((string) $qtyCarton, (string) $qtyPiece, (string) $cart[$line]['unit_price_bulk'], (string) $cart[$line]['price_per_piece'], $quantityPieces, (string) ($cart[$line]['discount'] ?? '0'), (int) ($cart[$line]['discount_type'] ?? 0), true);
                            $this->set_cart($cart);
                        }
                    }
                } else {
                    $this->add_item($row->item_id, (int) round($qp), $row->item_location, $row->discount ?? 0, $row->discount_type ?? 0, $row->item_unit_price, $row->description ?? '', $row->serialnumber ?? '', 1.0, $receiving_id, true, $expireDate);
                }
            }
        }

        $this->set_supplier((int) $this->receiving->get_supplier($receiving_id)->person_id);
    }

    /**
     * @return void
     */
    public function clear_all(): void
    {
        $this->clear_mode();
        $this->empty_cart();
        $this->remove_supplier();
        $this->clear_comment();
        $this->clear_reference();
        $this->clear_receiving_id();
        $this->clear_payment_type();
    }

    /**
     * Load an existing receiving for edit in the register. Copies items, supplier, and metadata.
     *
     * @param int $receiving_id
     * @return void
     */
    public function load_receiving_for_edit(int $receiving_id): void
    {
        $this->clear_all();
        $this->copy_entire_receiving($receiving_id);

        $info = $this->receiving->get_info($receiving_id)->getRow();
        if ($info) {
            $this->set_reference($info->reference ?? '');
            $this->set_comment($info->comment ?? '');
            $this->set_payment_type($info->payment_type ?? null);
        }
        $this->set_receiving_id($receiving_id);
    }

    /**
     * @param float $quantity
     * @param float $price
     * @param float $discount
     * @param int|null $discount_type
     * @param float $receiving_quantity
     * @return string
     */
    public function get_item_total(float $quantity, float $price, float $discount, ?int $discount_type, float $receiving_quantity): string
    {
        $extended_quantity = bcmul((string) $quantity, (string) $receiving_quantity, quantity_decimals());
        $total = bcmul($extended_quantity, (string) $price, totals_decimals());
        $dt = (int) ($discount_type ?? 0);
        $discount_amount = $this->get_item_discount($extended_quantity, (string) $price, (string) $discount, $dt);

        return bcsub($total, $discount_amount, totals_decimals());
    }

    /**
     * Get total pieces/quantity received from a cart item for inventory. Handles bulk items (qty_carton + qty_piece).
     *
     * @param array $item Cart item with quantity, receiving_quantity, and optionally qty_carton, qty_piece
     * @return float Total pieces for inventory update
     */
    public function get_items_received_from_cart_item(array $item): float
    {
        $rq = (float) ($item['receiving_quantity'] ?? 1);
        if (isset($item['qty_carton'], $item['qty_piece']) && $rq > 0) {
            $cartons = (float) ($item['qty_carton'] ?? 0);
            $pieces = (float) ($item['qty_piece'] ?? 0);

            return $cartons * $rq + $pieces;
        }
        $qty = (float) ($item['quantity'] ?? 0);

        return $rq != 0 ? $qty * $rq : $qty;
    }

    /**
     * Get item total from cart entry. Uses bulk formula when item has qty_carton/qty_piece.
     */
    public function get_item_total_for_cart(array $item, bool $include_discount = false): string
    {
        $q = (string) ($item['quantity'] ?? '0');
        $p = (string) ($item['price'] ?? '0');
        $d = (string) ($item['discount'] ?? '0');
        $dt = (int) ($item['discount_type'] ?? 0);
        $rq = (float) ($item['receiving_quantity'] ?? 1);
        if (isset($item['qty_carton'], $item['qty_piece'], $item['unit_price_bulk'], $item['price_per_piece'])) {
            $quantity_pieces = bcadd(
                bcmul((string) $item['qty_carton'], (string) $rq, quantity_decimals()),
                (string) $item['qty_piece'],
                quantity_decimals()
            );

            return $this->get_item_total_bulk(
                (string) $item['qty_carton'],
                (string) $item['qty_piece'],
                (string) $item['unit_price_bulk'],
                (string) $item['price_per_piece'],
                $quantity_pieces,
                $d,
                $dt,
                $include_discount
            );
        }

        $extended = bcmul($q, (string) $rq, quantity_decimals());
        $total = bcmul($extended, $p, totals_decimals());
        if ($include_discount) {
            $discount_amount = $this->get_item_discount($extended, $p, $d, $dt);

            return bcsub($total, $discount_amount, totals_decimals());
        }

        return $total;
    }

    /**
     * Compute line total: (qty_carton × unit_price_bulk) + (qty_piece × price_per_piece) minus discount.
     *
     * @param string $quantity_pieces Total pieces for discount calculation
     */
    public function get_item_total_bulk(string $qty_carton, string $qty_piece, string $unit_price_bulk, string $price_per_piece, string $quantity_pieces, string $discount, int $discount_type, bool $include_discount = false): string
    {
        $carton_total = bcmul($qty_carton, $unit_price_bulk, totals_decimals());
        $pieces_total = bcmul($qty_piece, $price_per_piece, totals_decimals());
        $total = bcadd($carton_total, $pieces_total, totals_decimals());
        if ($include_discount) {
            $discount_amount = $this->get_item_discount($quantity_pieces, $price_per_piece, $discount, $discount_type);

            return bcsub($total, $discount_amount, totals_decimals());
        }

        return $total;
    }

    /**
     * @param string $quantity
     * @param string $price
     * @param string $discount
     * @param int $discount_type
     * @return string
     */
    public function get_item_discount(string $quantity, string $price, string $discount, int $discount_type): string
    {
        $total = bcmul($quantity, $price, totals_decimals());
        if ($discount_type == PERCENT) {
            $discount = bcmul($total, bcdiv($discount, '100', totals_decimals() + 2), totals_decimals());
        } else {
            $discount = bcmul($quantity, $discount, totals_decimals());
        }

        return (string) round((float) $discount, totals_decimals(), PHP_ROUND_HALF_UP);
    }

    /**
     * @return string
     */
    public function get_total(): string
    {
        $total = '0';
        foreach ($this->get_cart() as $item) {
            $total = bcadd($total, $this->get_item_total_for_cart($item, true), totals_decimals());
        }

        return $total;
    }
}
