<?php

namespace App\Models\Reports;

use App\Models\Receiving;

/**
 *
 *
 * @property receiving receiving
 *
 */
class Detailed_receivings extends Report
{
    /**
     * @param array $inputs
     * @return void
     */
    public function create(array $inputs): void
    {
        // Create our temp tables to work with the data in our report
        $receiving = model(Receiving::class);
        $receiving->create_temp_table($inputs);
    }

    /**
     * @return array
     */
    public function getDataColumns(): array
    {
        return [
            'summary' => [
                ['id'             => lang('Reports.receiving_id')],
                ['receiving_time' => lang('Reports.date'), 'sortable' => false],
                ['quantity'       => lang('Reports.quantity')],
                ['employee_name'  => lang('Reports.received_by')],
                ['supplier_name'  => lang('Reports.supplied_by')],
                ['total'          => lang('Reports.total'), 'sorter' => 'number_sorter'],
                ['payment_type'   => lang('Reports.payment_type')],
                ['comment'        => lang('Reports.comments')],
                ['reference'      => lang('Receivings.reference')]
            ],
            'details' => [
                lang('Reports.item_number'),
                lang('Reports.name'),
                lang('Reports.category'),
                lang('Reports.quantity'),
                lang('Sales.carton_qty'),
                lang('Sales.pieces_qty'),
                lang('Reports.total'),
                lang('Reports.discount'),
                lang('Items.expire_date')
            ]
        ];
    }

    /**
     * @param string $receiving_id
     * @return array
     */
    public function getDataByReceivingId(string $receiving_id): array
    {
        $builder = $this->db->table('receivings_items_temp');
        $builder->select('receiving_id,
            MAX(receiving_time) as receiving_time,
            SUM(quantity_purchased) AS items_purchased,
            MAX(CONCAT(employee.first_name, " ", employee.last_name)) AS employee_name,
            MAX(supplier.company_name) AS supplier_name,
            SUM(subtotal) AS subtotal,
            SUM(total) AS total,
            SUM(profit) AS profit,
            MAX(payment_type) as payment_type,
            MAX(comment) as comment,
            MAX(reference) as reference');
        $builder->join('people AS employee', 'receivings_items_temp.employee_id = employee.person_id');
        $builder->join('suppliers AS supplier', 'receivings_items_temp.supplier_id = supplier.person_id', 'left');
        $builder->where('receiving_id', $receiving_id);
        $builder->groupBy('receiving_id');

        return $builder->get()->getRowArray();
    }

    /**
     * @param array $inputs
     * @return array
     */
    public function getData(array $inputs): array
    {
        $builder = $this->db->table('receivings_items_temp AS receivings_items_temp');
        $builder->select('receiving_id,
            MAX(receiving_time) as receiving_time,
            SUM(quantity_purchased) AS items_purchased,
            MAX(CONCAT(employee.first_name," ",employee.last_name)) AS employee_name,
            MAX(supplier.company_name) AS supplier_name,
            SUM(total) AS total,
            SUM(profit) AS profit,
            MAX(payment_type) AS payment_type,
            MAX(comment) AS comment,
            MAX(reference) AS reference');
        $builder->join('people AS employee', 'receivings_items_temp.employee_id = employee.person_id');
        $builder->join('suppliers AS supplier', 'receivings_items_temp.supplier_id = supplier.person_id', 'left');

        if ($inputs['location_id'] != 'all') {
            $builder->where('item_location', $inputs['location_id']);
        }

        if ($inputs['receiving_type'] == 'receiving') {    // TODO: These if statements should be replaced with a switch statement
            $builder->where('quantity_purchased >', 0);
        } elseif ($inputs['receiving_type'] == 'returns') {
            $builder->where('quantity_purchased <', 0);
        } elseif ($inputs['receiving_type'] == 'requisitions') {
            $builder->having('items_purchased = 0');
        }

        $builder->groupBy('receiving_id', 'receiving_time');
        $builder->orderBy('MAX(receiving_id)');

        $data = [];
        $data['summary'] = $builder->get()->getResultArray();
        $data['details'] = [];

        $builder = $this->db->table('receivings_items_temp AS rit');

        foreach ($data['summary'] as $key => $value) {
            $builder->select('
                MAX(report_items.name) AS name,
                MAX(report_items.item_number) AS item_number,
                MAX(report_items.category) AS category,
                MAX(rit.quantity_purchased) AS quantity_purchased,
                MAX(rit.serialnumber) AS serialnumber,
                MAX(rit.total) AS total,
                MAX(rit.discount) AS discount,
                MAX(rit.discount_type) AS discount_type,
                MAX(rit.item_location) AS item_location,
                MAX(rit.item_receiving_quantity) AS receiving_quantity,
                IFNULL(GROUP_CONCAT(DISTINCT rit.expire_date ORDER BY rit.line SEPARATOR ", "), "") AS expire_date');
            $builder->join('items AS report_items', 'rit.item_id = report_items.item_id');
            if ($this->db->fieldExists('item_unit', $this->db->getPrefix() . 'items')) {
                $builder->select('MAX(report_items.item_unit) AS item_unit, MAX(report_items.qty_per_unit) AS qty_per_unit, MAX(report_items.qty_per_carton) AS qty_per_carton, MAX(report_items.qty_per_box) AS qty_per_box, MAX(report_items.qty_per_bag) AS qty_per_bag, MAX(report_items.qty_per_pack) AS qty_per_pack, MAX(report_items.pack_name) AS pack_name');
            }

            if (count($inputs['definition_ids']) > 0) {
                $format = $this->db->escape(dateformat_mysql());
                $builder->select('GROUP_CONCAT(DISTINCT CONCAT_WS(\'_\', definition_id, attribute_value) ORDER BY definition_id SEPARATOR \'|\') AS attribute_values');
                $builder->select("GROUP_CONCAT(DISTINCT CONCAT_WS('_', definition_id, DATE_FORMAT(attribute_date, $format)) SEPARATOR '|') AS attribute_dtvalues");
                $builder->select('GROUP_CONCAT(DISTINCT CONCAT_WS(\'_\', definition_id, attribute_decimal) SEPARATOR \'|\') AS attribute_dvalues');
                $builder->join('attribute_links', 'attribute_links.item_id = report_items.item_id AND attribute_links.receiving_id = rit.receiving_id AND definition_id IN (' . implode(',', $inputs['definition_ids']) . ')', 'left');
                $builder->join('attribute_values', 'attribute_values.attribute_id = attribute_links.attribute_id', 'left');
            }

            $builder->where('rit.receiving_id', $value['receiving_id']);
            $builder->groupBy('rit.receiving_id, rit.item_id');
            $data['details'][$key] = $builder->get()->getResultArray();
        }

        return $data;
    }

    /**
     * @param array $inputs
     * @return array
     */
    public function getSummaryData(array $inputs): array
    {
        $builder = $this->db->table('receivings_items_temp');
        $builder->select('SUM(total) AS total');

        if ($inputs['location_id'] != 'all') {
            $builder->where('item_location', $inputs['location_id']);
        }

        switch ($inputs['receiving_type']) {
            case 'receiving':
                $builder->where('quantity_purchased >', 0);
                break;

            case 'returns':
                $builder->where('quantity_purchased <', 0);
                break;

            case 'requisitions':
                $builder->where('quantity_purchased', 0);
                break;
        }

        return $builder->get()->getRowArray();
    }
}
