# Recreate the production folder for deployment
# Run from project root: .\create_production.ps1

$ErrorActionPreference = 'Stop'
$root = $PSScriptRoot

if (Test-Path "$root\production") {
    Remove-Item -Recurse -Force "$root\production"
}
New-Item -ItemType Directory -Path "$root\production" -Force | Out-Null

Write-Host "Copying app, public, vendor..."
robocopy "$root\app" "$root\production\app" /E /NFL /NDL /NJH /NJS | Out-Null
robocopy "$root\public" "$root\production\public" /E /NFL /NDL /NJH /NJS | Out-Null
robocopy "$root\vendor" "$root\production\vendor" /E /NFL /NDL /NJH /NJS | Out-Null

Write-Host "Copying essential files..."
Copy-Item "$root\spark" "$root\production\" -Force
Copy-Item "$root\.env.production" "$root\production\.env" -Force
Copy-Item "$root\DEPLOYMENT.md" "$root\production\" -Force
Copy-Item "$root\INSTALL.md" "$root\production\" -Force
Copy-Item "$root\README.md" "$root\production\" -Force

Write-Host "Creating writable structure..."
$dirs = @("$root\production\writable\cache", "$root\production\writable\logs",
    "$root\production\writable\session", "$root\production\writable\uploads",
    "$root\production\writable\backups", "$root\production\writable\backup")
foreach ($d in $dirs) { New-Item -ItemType Directory -Path $d -Force | Out-Null }
Copy-Item "$root\writable\.htaccess" "$root\production\writable\" -Force
Copy-Item "$root\writable\index.html" "$root\production\writable\" -Force
Copy-Item "$root\writable\cache\.htaccess" "$root\production\writable\cache\" -Force
Copy-Item "$root\writable\cache\index.html" "$root\production\writable\cache\" -Force
Copy-Item "$root\writable\logs\index.html" "$root\production\writable\logs\" -Force
Copy-Item "$root\writable\session\index.html" "$root\production\writable\session\" -Force
Copy-Item "$root\writable\uploads\index.html" "$root\production\writable\uploads\" -Force
Copy-Item "$root\writable\backup\index.html" "$root\production\writable\backup\" -Force

Write-Host "Adding protection and instructions..."
@"
<IfModule mod_authz_core.c>
    Require all denied
</IfModule>
"@ | Set-Content "$root\production\.htaccess" -NoNewline
$uploadReadme = @"
OSPOS Production Package - Upload Instructions
==============================================

CONTENTS
--------
- app/          Application code
- public/       Web document root (point your server here)
- vendor/       PHP dependencies
- writable/     Logs, cache, session, uploads (must be writable)
- .env          Configuration (edit with your database credentials)
- spark         CLI tool
- DEPLOYMENT.md Full deployment guide

QUICK DEPLOYMENT
----------------
1. Upload this entire folder to your server
2. Point your web server document root to the "public" subfolder
3. Edit .env with your database credentials and base URL
4. Run: php spark key:generate  (copy key to .env encryption.key)
5. Import database: mysql -u user -p dbname < app/Database/database.sql
6. Set writable/ and subfolders to be writable by web server (chmod 755 or 775)

IMPORTANT: Only public/ should be in document root. See DEPLOYMENT.md for full instructions.
"@
$uploadReadme | Set-Content "$root\production\UPLOAD_README.txt"

Write-Host "Done! Production folder ready at $root\production"
Write-Host "Zip it for upload: Compress-Archive -Path $root\production\* -DestinationPath $root\ospos_production.zip"
