<?php

namespace Config;

use CodeIgniter\Config\BaseConfig;

/**
 * GTS APPSTORE License API configuration.
 * Override via .env: license.apiUrl, license.appId, license.apiSecret, license.cacheTtl, license.failOpen, license.timeout
 */
class License extends BaseConfig
{
    /**
     * API endpoint for license verification
     */
    public string $apiUrl = 'https://gtsappstore.com/api/verify-license';

    /**
     * Application identifier - use "gts-pos" or get from GET https://gtsappstore.com/api/applications
     */
    public string $appId = 'gts-pos';

    /**
     * Secret for HMAC-SHA256 signature verification. Get from GTS APPSTORE admin.
     * Leave empty to skip signature verification. If set but verification fails
     * (e.g. encoding mismatch), set skipSignatureVerification = true to proceed.
     */
    public string $apiSecret = '';

    /**
     * Skip signature verification even when apiSecret is set.
     * Use when the API's JSON encoding differs from ours. Not recommended for production.
     */
    public bool $skipSignatureVerification = false;

    /**
     * Cache TTL in seconds (1-12 hours). Default 4 hours.
     */
    public int $cacheTtl = 14400;

    /**
     * When API is unreachable: true = allow access temporarily (fail open), false = deny access (fail closed)
     */
    public bool $failOpen = false;

    /**
     * Request timeout in seconds
     */
    public int $timeout = 10;

    /**
     * Enable license verification. Set to false to bypass in development only.
     * In production, license check always runs regardless of this setting.
     */
    public bool $enabled = true;

    public function __construct()
    {
        parent::__construct();

        $this->apiUrl    = env('license.apiUrl', $this->apiUrl);
        $this->appId     = env('license.appId', $this->appId);
        $this->apiSecret                 = env('license.apiSecret', $this->apiSecret);
        $this->skipSignatureVerification = filter_var(env('license.skipSignatureVerification', 'false'), FILTER_VALIDATE_BOOLEAN);
        $this->cacheTtl                  = (int) env('license.cacheTtl', (string) $this->cacheTtl);
        $this->failOpen  = filter_var(env('license.failOpen', 'false'), FILTER_VALIDATE_BOOLEAN);
        $this->timeout   = (int) env('license.timeout', (string) $this->timeout);
        $this->enabled   = filter_var(env('license.enabled', 'true'), FILTER_VALIDATE_BOOLEAN);
    }
}
