<?php

namespace Config;

use CodeIgniter\Config\BaseConfig;

/**
 * Menu categories for grouping navigation modules.
 * Maps module_id to category. Modules not in this map appear as top-level items.
 *
 * Category order and labels are defined in the lang file Config.menu_category_*
 */
class MenuCategories extends BaseConfig
{
    /**
     * Maps module_id to category key. Category keys are used for lang keys: Config.menu_category_{key}
     *
     * @var array<string, string>
     */
    public array $categories = [
        'sales'              => 'operations',
        'receivings'         => 'operations',
        'transfers'          => 'operations',
        'items'              => 'catalog',
        'item_kits'          => 'catalog',
        'giftcards'          => 'catalog',
        'attributes'         => 'catalog',
        'customers'          => 'people',
        'suppliers'          => 'people',
        'employees'          => 'people',
        'reports'             => 'finance',
        'cashups'             => 'finance',
        'admin_cashups'       => 'finance',
        'expenses'            => 'finance',
        'expenses_categories' => 'finance',
        'taxes'               => 'finance',
        'config'              => 'admin',
        'migrate'             => 'admin',
        'messages'             => 'admin',
    ];

    /**
     * Minimum number of items in a category to show as dropdown. Below this, show as flat items.
     */
    public int $dropdownThreshold = 2;

    /**
     * Menu icon mapping: module_id => Bootstrap 3 Glyphicon class (without glyphicon- prefix).
     * Use 'svg' to fall back to images/menubar/{module_id}.svg
     *
     * @var array<string, string>
     */
    public array $menuIcons = [
        'sales'              => 'shopping-cart',
        'receivings'         => 'import',
        'transfers'          => 'transfer',
        'items'              => 'tags',
        'item_kits'          => 'th-large',
        'giftcards'          => 'gift',
        'attributes'         => 'list-alt',
        'customers'          => 'user',
        'suppliers'          => 'briefcase',
        'employees'          => 'user',
        'reports'            => 'stats',
        'cashups'            => 'usd',
        'admin_cashups'      => 'usd',
        'expenses'           => 'minus-sign',
        'expenses_categories' => 'folder-open',
        'taxes'               => 'education',
        'config'             => 'cog',
        'migrate'            => 'refresh',
        'messages'           => 'envelope',
        'home'               => 'home',
        'office'             => 'blackboard',
    ];

    /**
     * Get the icon for a module. Returns glyphicon class or 'svg' for image fallback.
     */
    public function getMenuIcon(string $moduleId): string
    {
        return $this->menuIcons[$moduleId] ?? 'svg';
    }

    /**
     * Get the category for a module, or null if ungrouped.
     */
    public function getCategory(string $moduleId): ?string
    {
        return $this->categories[$moduleId] ?? null;
    }

    /**
     * Group modules by category for nav display.
     *
     * @param object[] $modules
     * @return array{groups: array<string, object[]>, ungrouped: object[]}
     */
    public function groupModules(array $modules): array
    {
        $groups = [];
        $ungrouped = [];

        foreach ($modules as $module) {
            $category = $this->getCategory($module->module_id);
            if ($category !== null) {
                $groups[$category][] = $module;
            } else {
                $ungrouped[] = $module;
            }
        }

        return ['groups' => $groups, 'ungrouped' => $ungrouped];
    }
}
