<?php

namespace App\Controllers;

use App\Libraries\LicenseService;
use CodeIgniter\HTTP\RedirectResponse;
use CodeIgniter\HTTP\ResponseInterface;
use Config\OSPOS;

/**
 * License activation controller.
 * Allows users to enter and verify their GTS APPSTORE license key.
 */
class License extends BaseController
{
    public function activate(): string
    {
        $licenseService = new LicenseService();
        try {
            $config = config(OSPOS::class)->settings ?? [];
        } catch (\Throwable) {
            $config = [];
        }
        $theme          = $config['theme'] ?? 'flatly';
        if (in_array($theme, ['paper', 'readable'], true)) {
            $theme = 'flatly';
        }
        $company = $config['company'] ?? 'GTS Point of Sale';

        $data = [
            'config'          => $config,
            'theme'           => $theme,
            'company'         => $company,
            'license_key'     => $licenseService->getLicenseKey(),
            'message'         => $this->request->getGet('message') ?? session('license_error'),
            'status'          => $this->request->getGet('status') ?? session('license_status'),
            'license_service' => $licenseService,
        ];

        return view('license/activate', $data);
    }

    /**
     * Verify license via API and save if valid.
     * Returns JSON for AJAX requests, redirects otherwise.
     */
    public function verify(): RedirectResponse|ResponseInterface
    {
        $licenseKey  = $this->request->getPost('license_key');
        $secretToken = $this->request->getPost('secret_token');
        $isAjax      = $this->request->isAJAX();

        if (empty(trim($licenseKey ?? ''))) {
            $msg = lang('License.invalid');
            return $isAjax
                ? $this->response->setJSON(['success' => false, 'message' => $msg])
                : redirect()->back()->withInput()->with('license_error', $msg);
        }

        $licenseService = new LicenseService();
        $licenseService->saveLicense([
            'license_key'  => $licenseKey,
            'secret_token' => $secretToken ?? '',
        ]);

        $result = $licenseService->verify(null, true);

        if ($result === null) {
            $msg = 'License verification service unavailable. Please try again later.';
            return $isAjax
                ? $this->response->setJSON(['success' => false, 'message' => $msg])
                : redirect()->back()->withInput()->with('license_error', $msg);
        }

        if (($result['valid'] ?? false) && ($result['status'] ?? '') === 'active') {
            session()->remove('license_error');
            session()->remove('license_status');

            $redirectTo = $this->request->getPost('redirect') ?: 'login';
            $msg        = lang('License.success');

            return $isAjax
                ? $this->response->setJSON([
                    'success'  => true,
                    'message'  => $msg,
                    'redirect' => base_url($redirectTo),
                ])
                : redirect()->to($redirectTo)->with('license_success', $msg);
        }

        $message = $result['message'] ?? lang('License.invalid');
        $status  = $result['status'] ?? 'invalid';

        return $isAjax
            ? $this->response->setJSON(['success' => false, 'message' => $message])
            : redirect()
                ->back()
                ->withInput()
                ->with('license_error', $message)
                ->with('license_status', $status);
    }
}
