<?php

namespace App\Controllers;

use App\Libraries\Receiving_lib;
use App\Libraries\Token_lib;
use App\Libraries\Barcode_lib;
use App\Models\Inventory;
use App\Models\Item;
use App\Models\Item_kit;
use App\Models\Receiving;
use App\Models\Stock_location;
use App\Models\Supplier;
use Config\OSPOS;
use Config\Services;
use ReflectionException;

class Receivings extends Secure_Controller
{
    private Receiving_lib $receiving_lib;
    private Token_lib $token_lib;
    private Barcode_lib $barcode_lib;
    private Inventory $inventory;
    private Item $item;
    private Item_kit $item_kit;
    private Receiving $receiving;
    private Stock_location $stock_location;
    private Supplier $supplier;
    private array $config;

    public function __construct()
    {
        parent::__construct('receivings');

        $this->receiving_lib = new Receiving_lib();
        $this->token_lib = new Token_lib();
        $this->barcode_lib = new Barcode_lib();

        $this->inventory = model(Inventory::class);
        $this->item_kit = model(Item_kit::class);
        $this->item = model(Item::class);
        $this->receiving = model(Receiving::class);
        $this->stock_location = model(Stock_location::class);
        $this->supplier = model(Supplier::class);
        $this->config = config(OSPOS::class)->settings;
    }

    /**
     * @return void
     */
    public function getIndex(): void
    {
        $this->_reload();
    }

    /**
     * Returns search suggestions for an item. Used in app/Views/sales/register.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function getItemSearch(): void
    {
        $search = $this->request->getGet('term');
        $suggestions = $this->item->get_search_suggestions($search, ['search_custom' => false, 'is_deleted' => false], true);
        $suggestions = array_merge($suggestions, $this->item_kit->get_search_suggestions($search));

        echo json_encode($suggestions);
    }

    /**
     * Gets search suggestions for a stock item. Used in app/Views/receivings/receiving.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function getStockItemSearch(): void
    {
        $search = $this->request->getGet('term');
        $suggestions = $this->item->get_stock_search_suggestions($search, ['search_custom' => false, 'is_deleted' => false], true);
        $suggestions = array_merge($suggestions, $this->item_kit->get_search_suggestions($search));

        echo json_encode($suggestions);
    }

    /**
     * Set supplier if it exists in the database. Used in app/Views/receivings/receiving.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function postSelectSupplier(): void
    {
        $supplier_id = $this->request->getPost('supplier', FILTER_SANITIZE_NUMBER_INT);
        if ($this->supplier->exists($supplier_id)) {
            $this->receiving_lib->set_supplier($supplier_id);
        }

        $this->_reload();    // TODO: Hungarian notation
    }

    /**
     * Change receiving mode for current receiving. Used in app/Views/receivings/receiving.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function postChangeMode(): void
    {
        $stock_destination = $this->request->getPost('stock_destination', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $stock_source = $this->request->getPost('stock_source', FILTER_SANITIZE_NUMBER_INT);

        if ((!$stock_source || $stock_source == $this->receiving_lib->get_stock_source()) &&
            (!$stock_destination || $stock_destination == $this->receiving_lib->get_stock_destination())
        ) {
            $this->receiving_lib->clear_reference();
            $mode = $this->request->getPost('mode', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
            $this->receiving_lib->set_mode($mode);
        } elseif ($this->stock_location->is_allowed_location($stock_source, 'receivings')) {
            $this->receiving_lib->set_stock_source($stock_source);
            $this->receiving_lib->set_stock_destination($stock_destination);
        }

        $this->_reload();    // TODO: Hungarian notation
    }

    /**
     * Sets receiving comment. Used in app/Views/receivings/receiving.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function postSetComment(): void
    {
        $this->receiving_lib->set_comment($this->request->getPost('comment', FILTER_SANITIZE_FULL_SPECIAL_CHARS));
    }

    /**
     * Sets the print after sale flag for the receiving. Used in app/Views/receivings/receiving.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function postSetPrintAfterSale(): void
    {
        $this->receiving_lib->set_print_after_sale($this->request->getPost('recv_print_after_sale') != null);
    }

    /**
     * Sets the reference number for the receiving.  Used in app/Views/receivings/receiving.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function postSetReference(): void
    {
        $this->receiving_lib->set_reference($this->request->getPost('recv_reference', FILTER_SANITIZE_FULL_SPECIAL_CHARS));
    }

    /**
     * Add an item to the receiving. Used in app/Views/receivings/receiving.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function postAdd(): void
    {
        $data = [];

        $mode = $this->receiving_lib->get_mode();
        $item_id_or_number_or_item_kit_or_receipt = $this->request->getPost('item', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $this->token_lib->parse_barcode($quantity, $price, $item_id_or_number_or_item_kit_or_receipt);
        $quantity = ($mode == 'receive' || $mode == 'requisition') ? $quantity : -$quantity;
        $item_location = $this->receiving_lib->get_stock_source();
        $discount = $this->config['default_receivings_discount'];
        $discount_type = $this->config['default_receivings_discount_type'];

        if ($mode == 'return' && $this->receiving->is_valid_receipt($item_id_or_number_or_item_kit_or_receipt)) {
            $this->receiving_lib->return_entire_receiving($item_id_or_number_or_item_kit_or_receipt);
        } elseif ($this->item_kit->is_valid_item_kit($item_id_or_number_or_item_kit_or_receipt)) {
            $this->receiving_lib->add_item_kit($item_id_or_number_or_item_kit_or_receipt, $item_location, $discount, $discount_type);
        } elseif (!$this->receiving_lib->add_item($item_id_or_number_or_item_kit_or_receipt, $quantity, $item_location, $discount,  $discount_type)) {
            $data['error'] = lang('Receivings.unable_to_add_item');
        }

        $this->_reload($data);    // TODO: Hungarian notation
    }

    /**
     * Daily Receivings - view received items (like Daily Sales for receivings)
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function getManage(): void
    {
        $person_id = $this->session->get('person_id');

        if (!$this->employee->has_grant('reports_receivings', $person_id)) {
            redirect('no_access/receivings/reports_receivings');
        }

        $data['table_headers'] = get_receivings_manage_table_headers();
        $data['controller_name'] = 'receivings';

        echo view('receivings/manage', $data);
    }

    /**
     * Search receivings for the manage view (JSON)
     *
     * @return \CodeIgniter\HTTP\ResponseInterface
     * @noinspection PhpUnused
     */
    public function getSearch()
    {
        try {
            $search = $this->request->getGet('search', FILTER_SANITIZE_FULL_SPECIAL_CHARS) ?? '';
            $limit = $this->request->getGet('limit', FILTER_SANITIZE_NUMBER_INT) ?: 25;
            $offset = $this->request->getGet('offset', FILTER_SANITIZE_NUMBER_INT) ?: 0;
            $sort = $this->sanitizeSortColumn(receiving_headers(), $this->request->getGet('sort', FILTER_SANITIZE_FULL_SPECIAL_CHARS), 'receiving_id');
            $order = $this->request->getGet('order', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

            $config = config(OSPOS::class)->settings;
            $use_datetime = !empty($config['date_or_time_format']);
            $start_date = $this->request->getGet('start_date', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
            $end_date = $this->request->getGet('end_date', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
            if (empty($start_date) || empty($end_date)) {
                if ($use_datetime) {
                    $start_date = $start_date ?: date('Y-m-d H:i:s', strtotime('2010-01-01'));
                    $end_date = $end_date ?: date('Y-m-d 23:59:59');
                } else {
                    $start_date = $start_date ?: date('Y-m-d', strtotime('2010-01-01'));
                    $end_date = $end_date ?: date('Y-m-d');
                }
            }
            $filters = ['start_date' => $start_date, 'end_date' => $end_date];

            $receivings = $this->receiving->search($search, $filters, $limit, $offset, $sort, $order);
            $total_rows = $this->receiving->get_found_rows($search, $filters);

            $data_rows = [];
            foreach ($receivings->getResult() as $receiving) {
                $data_rows[] = get_receiving_data_row($receiving);
            }

            if ($total_rows > 0) {
                $data_rows[] = get_receiving_data_last_row($receivings);
            }

            return $this->response->setJSON(['total' => $total_rows, 'rows' => $data_rows]);
        } catch (\Throwable $e) {
            log_message('error', 'Receivings getSearch: ' . $e->getMessage());
            return $this->response->setJSON(['total' => 0, 'rows' => [], 'error' => $e->getMessage()]);
        }
    }

    /**
     * Get single receiving row (for bootstrap-table)
     *
     * @param int $receiving_id
     * @return void
     * @noinspection PhpUnused
     */
    public function getRow(int $receiving_id): void
    {
        $filters = [
            'start_date'   => '1970-01-01',
            'end_date'     => date('Y-m-d', strtotime('+1 year')),
            'receiving_id' => $receiving_id
        ];
        $receivings = $this->receiving->search('RECV ' . $receiving_id, $filters, 1, 0, 'receiving_id', 'asc');
        $rows = $receivings->getResult();
        if (!empty($rows)) {
            echo json_encode(get_receiving_data_row($rows[0]));
        } else {
            echo json_encode([]);
        }
    }

    /**
     * Edit line item in current receiving. Used in app/Views/receivings/receiving.php
     *
     * @param $item_id
     * @return void
     * @noinspection PhpUnused
     */
    public function postEditItem($item_id): void
    {
        $data = [];

        $validation_rule = [
            'price'    => 'trim|required|decimal_locale',
            'quantity' => 'trim|permit_empty|decimal_locale',
            'qty_carton' => 'trim|permit_empty|decimal_locale',
            'qty_piece'  => 'trim|permit_empty|decimal_locale',
            'discount' => 'trim|permit_empty|decimal_locale',
        ];

        $price = parse_decimals($this->request->getPost('price'));
        $raw_receiving_quantity = parse_quantity($this->request->getPost('receiving_quantity'));
        $receiving_quantity = (float) filter_var($raw_receiving_quantity, FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION) ?: 1;

        $qty_carton_str = null;
        $qty_piece_str = null;
        $unit_price_bulk_str = null;
        $price_per_piece_str = null;

        if ($receiving_quantity > 1) {
            $qty_carton = parse_decimals($this->request->getPost('qty_carton') ?? '0') ?: 0;
            $qty_piece = parse_decimals($this->request->getPost('qty_piece') ?? '0') ?: 0;
            $quantity = $qty_carton * $receiving_quantity + $qty_piece;
            $qty_carton_str = (string) $qty_carton;
            $qty_piece_str = (string) $qty_piece;
            $unit_price_bulk_str = (string) $price;
            // Use 6 decimals to avoid rounding error: e.g. 200/24=8.333333 preserves 200 on display (8.333333*24=199.999992≈200)
            $price_per_piece_str = $receiving_quantity > 0 ? bcdiv((string) $price, (string) $receiving_quantity, 6) : (string) $price;
            $price = (float) $price_per_piece_str;
        } else {
            $quantity = parse_quantity($this->request->getPost('quantity')) ?: 0;
        }

        $description = $this->request->getPost('description', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $serialnumber = $this->request->getPost('serialnumber', FILTER_SANITIZE_FULL_SPECIAL_CHARS) ?? '';
        $discount_type = $this->request->getPost('discount_type', FILTER_SANITIZE_NUMBER_INT);
        $discount = $discount_type
            ? parse_quantity(filter_var($this->request->getPost('discount'), FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION))
            : parse_decimals(filter_var($this->request->getPost('discount'), FILTER_SANITIZE_NUMBER_FLOAT, FILTER_FLAG_ALLOW_FRACTION));

        $expire_date_raw = $this->request->getPost('expire_date', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $expire_date = null;
        if (!empty($expire_date_raw)) {
            $parsed = strtotime($expire_date_raw);
            $expire_date = $parsed ? date('Y-m-d', $parsed) : null;
        }

        $batch_number = $this->request->getPost('batch_number', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $batch_number = !empty($batch_number) ? trim($batch_number) : null;

        $rules = $validation_rule;
        if ($receiving_quantity <= 1) {
            $rules['quantity'] = 'trim|required|decimal_locale';
        }
        $valid = $this->validate($rules);

        if ($valid) {
            $this->receiving_lib->edit_item($item_id, $description, $serialnumber, $quantity, $discount, $discount_type, $price, $receiving_quantity, $qty_carton_str, $qty_piece_str, $unit_price_bulk_str, $price_per_piece_str, $expire_date, $batch_number);
        } else {
            $data['error'] = lang('Receivings.error_editing_item');
        }

        $this->_reload($data);
    }

    /**
     * Edit a receiving. Loads the receiving into the register and redirects so the user can edit and complete.
     *
     * @param int $receiving_id
     * @return \CodeIgniter\HTTP\RedirectResponse|void
     * @noinspection PhpUnused
     */
    public function getEdit(int $receiving_id)
    {
        $this->receiving_lib->load_receiving_for_edit($receiving_id);

        return redirect()->to(site_url('receivings'));
    }

    /**
     * Deletes an item from the current receiving. Used in app/Views/receivings/receiving.php
     *
     * @param $item_number
     * @return void
     * @noinspection PhpUnused
     */
    public function getDeleteItem($item_number): void
    {
        $this->receiving_lib->delete_item($item_number);

        $this->_reload();    // TODO: Hungarian notation
    }

    /**
     * @throws ReflectionException
     */
    public function postDelete(int $receiving_id = -1, bool $update_inventory = true): void
    {
        $employee_id = $this->employee->get_logged_in_employee_info()->person_id;
        $receiving_ids = $receiving_id == -1 ? $this->request->getPost('ids', FILTER_SANITIZE_NUMBER_INT) : [$receiving_id];    // TODO: Replace -1 with constant

        if ($this->receiving->delete_list($receiving_ids, $employee_id, $update_inventory)) {    // TODO: Likely need to surround this block of code in a try-catch to catch the ReflectionException
            echo json_encode([
                'success' => true,
                'message' => lang('Receivings.successfully_deleted') . ' ' . count($receiving_ids) . ' ' . lang('Receivings.one_or_multiple'),
                'ids'     => $receiving_ids
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => lang('Receivings.cannot_be_deleted')]);
        }
    }

    /**
     * Removes a supplier from a receiving. Used in app/Views/receivings/receiving.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function getRemoveSupplier(): void
    {
        $this->receiving_lib->clear_reference();
        $this->receiving_lib->remove_supplier();

        $this->_reload();    // TODO: Hungarian notation
    }

    /**
     * Complete and finalize receiving.  Used in app/Views/receivings/receiving.php
     *
     * @throws ReflectionException
     * @noinspection PhpUnused
     */
    public function postComplete(): void
    {

        $data = [];

        $data['cart'] = $this->receiving_lib->get_cart();
        $data['total'] = $this->receiving_lib->get_total();
        $data['transaction_time'] = to_datetime(time());
        $data['mode'] = $this->receiving_lib->get_mode();
        $data['comment'] = $this->receiving_lib->get_comment();
        $data['reference'] = $this->receiving_lib->get_reference();
        $data['payment_type'] = $this->request->getPost('payment_type', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $payment_reference = trim((string) ($this->request->getPost('payment_reference', FILTER_SANITIZE_FULL_SPECIAL_CHARS) ?? ''));
        $due_date_raw = $this->request->getPost('due_date', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $data['payment_reference'] = $payment_reference !== '' ? $payment_reference : null;
        $data['due_date'] = !empty($due_date_raw) && strtotime($due_date_raw) ? date('Y-m-d', strtotime($due_date_raw)) : null;
        $data['show_stock_locations'] = $this->stock_location->show_locations('receivings');
        $data['stock_location'] = $this->receiving_lib->get_stock_source();
        if ($this->request->getPost('amount_tendered') != null) {
            $data['amount_tendered'] = parse_decimals($this->request->getPost('amount_tendered'));
            $data['amount_change'] = to_currency($data['amount_tendered'] - $data['total']);
        }

        $employee_id = $this->employee->get_logged_in_employee_info()->person_id;
        $employee_info = $this->employee->get_info($employee_id);
        $data['employee'] = $employee_info->first_name . ' ' . $employee_info->last_name;

        $supplier_id = $this->receiving_lib->get_supplier();
        if ($supplier_id != -1) {
            $supplier_info = $this->supplier->get_info($supplier_id);
            $data['supplier'] = $supplier_info->company_name;    // TODO: duplicated code
            $data['first_name'] = $supplier_info->first_name;
            $data['last_name'] = $supplier_info->last_name;
            $data['supplier_email'] = $supplier_info->email;
            $data['supplier_address'] = $supplier_info->address_1;
            if (!empty($supplier_info->zip) or !empty($supplier_info->city)) {
                $data['supplier_location'] = $supplier_info->zip . ' ' . $supplier_info->city;
            } else {
                $data['supplier_location'] = '';
            }
        }

        $editing_id = $this->receiving_lib->get_receiving_id();

        if ($editing_id > 0) {
            // Update existing receiving
            $success = $this->receiving->update_value($editing_id, $data['cart'], $supplier_id, $employee_id, $data['comment'], $data['reference'], $data['payment_type']);

            if ($success) {
                $data['receiving_id'] = 'RECV ' . $editing_id;
                $data['barcode'] = $this->barcode_lib->generate_receipt_barcode($data['receiving_id']);
            } else {
                $data['receiving_id'] = 'RECV -1';
                $data['error_message'] = lang('Receivings.transaction_failed');
            }
        } else {
            // Save new receiving
            $data['receiving_id'] = 'RECV ' . $this->receiving->save_value($data['cart'], $supplier_id, $employee_id, $data['comment'], $data['reference'], $data['payment_type'], $data['stock_location'], $data['payment_reference'] ?? null, $data['due_date'] ?? null);

            if ($data['receiving_id'] == 'RECV -1') {
                $data['error_message'] = lang('Receivings.transaction_failed');
            } else {
                $data['barcode'] = $this->barcode_lib->generate_receipt_barcode($data['receiving_id']);
            }
        }

        $data['print_after_sale'] = $this->receiving_lib->is_print_after_sale();

        if (! isset($data['error_message'])) {
            $this->receiving_lib->clear_all();
        }
        echo view("receivings/receipt", $data);
    }

    /**
     * Complete a receiving requisition. Used in app/Views/receivings/receiving.php.
     *
     * @throws ReflectionException
     * @noinspection PhpUnused
     */
    public function postRequisitionComplete(): void
    {
        if ($this->receiving_lib->get_stock_source() != $this->receiving_lib->get_stock_destination()) {
            foreach ($this->receiving_lib->get_cart() as $item) {
                $this->receiving_lib->delete_item($item['line']);
                $this->receiving_lib->add_item($item['item_id'], $item['quantity'], $this->receiving_lib->get_stock_destination(), $item['discount_type']);
                $this->receiving_lib->add_item($item['item_id'], -$item['quantity'], $this->receiving_lib->get_stock_source(), $item['discount_type']);
            }

            $this->postComplete();
        } else {
            $data['error'] = lang('Receivings.error_requisition');

            $this->_reload($data);    // TODO: Hungarian notation
        }
    }

    /**
     * Gets the receipt for a receiving. Used in app/Views/receivings/form.php
     *
     * @param $receiving_id
     * @return void
     * @noinspection PhpUnused
     */
    public function getReceipt($receiving_id): void
    {
        $receiving_info = $this->receiving->get_info($receiving_id)->getRowArray();
        $this->receiving_lib->copy_entire_receiving($receiving_id);
        $data['cart'] = $this->receiving_lib->get_cart();
        $data['total'] = $this->receiving_lib->get_total();
        $data['mode'] = $this->receiving_lib->get_mode();
        $data['transaction_time'] = to_datetime(strtotime($receiving_info['receiving_time']));
        $data['show_stock_locations'] = $this->stock_location->show_locations('receivings');
        $data['payment_type'] = $receiving_info['payment_type'];
        $data['reference'] = $this->receiving_lib->get_reference();
        $data['receiving_id'] = 'RECV ' . $receiving_id;
        $data['barcode'] = $this->barcode_lib->generate_receipt_barcode($data['receiving_id']);
        $employee_info = $this->employee->get_info($receiving_info['employee_id']);
        $data['employee'] = $employee_info->first_name . ' ' . $employee_info->last_name;

        $supplier_id = $this->receiving_lib->get_supplier();    // TODO: Duplicated code
        if ($supplier_id != -1) {
            $supplier_info = $this->supplier->get_info($supplier_id);
            $data['supplier'] = $supplier_info->company_name;
            $data['first_name'] = $supplier_info->first_name;
            $data['last_name'] = $supplier_info->last_name;
            $data['supplier_email'] = $supplier_info->email;
            $data['supplier_address'] = $supplier_info->address_1;
            if (!empty($supplier_info->zip) or !empty($supplier_info->city)) {
                $data['supplier_location'] = $supplier_info->zip . ' ' . $supplier_info->city;
            } else {
                $data['supplier_location'] = '';
            }
        }

        $data['print_after_sale'] = false;

        echo view("receivings/receipt", $data);

        $this->receiving_lib->clear_all();
    }

    /**
     * @param array $data
     * @return void
     */
    private function _reload(array $data = []): void    // TODO: Hungarian notation
    {
        $cart = $this->receiving_lib->get_cart();
        foreach ($cart as $line => $item) {
            if (!isset($item['pic_filename'])) {
                $item_info = $this->item->get_info($item['item_id']);
                $cart[$line]['pic_filename'] = $item_info->pic_filename ?? null;
            }
            $item_info = $this->item->get_info($item['item_id']);
            $available_units = $this->item->get_available_units($item_info);
            $cart[$line]['available_units'] = $available_units;
            $receiving_qty = (float) ($item['receiving_quantity'] ?? 1);
            $qty = (float) ($item['quantity'] ?? 0);
            if (isset($item['qty_carton'], $item['qty_piece'])) {
                $cart[$line]['bulk_qty_display'] = (float) $item['qty_carton'];
                $cart[$line]['pieces_qty_display'] = (float) $item['qty_carton'] * $receiving_qty + (float) $item['qty_piece'];
            } else {
                $cart[$line]['bulk_qty_display'] = $receiving_qty > 1 ? $qty : 0;
                $cart[$line]['pieces_qty_display'] = $qty * $receiving_qty;
            }
            $bulk_unit = $item_info->item_unit ?? 'piece';
            $cart[$line]['bulk_label'] = ($bulk_unit !== 'piece' && isset($available_units[$bulk_unit]))
                ? ($available_units[$bulk_unit][1] ?? $bulk_unit) : lang('Items.unit_carton');
        }
        $data['cart'] = $cart;
        $data['modes'] = ['receive' => lang('Receivings.receiving'), 'return' => lang('Receivings.return')];
        $data['mode'] = $this->receiving_lib->get_mode();
        $data['stock_locations'] = $this->stock_location->get_allowed_locations('receivings');
        $data['show_stock_locations'] = count($data['stock_locations']) > 1;
        if ($data['show_stock_locations']) {
            $data['modes']['requisition'] = lang('Receivings.requisition');
            $data['stock_source'] = $this->receiving_lib->get_stock_source();
            $data['stock_destination'] = $this->receiving_lib->get_stock_destination();
        }

        $data['total'] = $this->receiving_lib->get_total();
        $data['items_module_allowed'] = $this->employee->has_grant('items', $this->employee->get_logged_in_employee_info()->person_id);
        $data['comment'] = $this->receiving_lib->get_comment();
        $data['reference'] = $this->receiving_lib->get_reference();
        $data['payment_options'] = $this->receiving->get_payment_options();
        $data['payment_type'] = $this->receiving_lib->get_payment_type();
        $data['editing_receiving_id'] = $this->receiving_lib->get_receiving_id();
        $data['payment_reference'] = '';
        $data['due_date'] = '';
        if ($data['editing_receiving_id'] > 0) {
            $recv_info = $this->receiving->get_info($data['editing_receiving_id'])->getRow();
            if ($recv_info) {
                $data['payment_reference'] = $recv_info->payment_reference ?? '';
                $data['due_date'] = $recv_info->due_date ?? '';
            }
        }

        $supplier_id = $this->receiving_lib->get_supplier();

        if ($supplier_id != -1) {    // TODO: Duplicated Code... replace -1 with a constant
            $supplier_info = $this->supplier->get_info($supplier_id);
            $data['supplier'] = $supplier_info->company_name;
            $data['first_name'] = $supplier_info->first_name;
            $data['last_name'] = $supplier_info->last_name;
            $data['supplier_email'] = $supplier_info->email;
            $data['supplier_address'] = $supplier_info->address_1;
            if (!empty($supplier_info->zip) or !empty($supplier_info->city)) {
                $data['supplier_location'] = $supplier_info->zip . ' ' . $supplier_info->city;
            } else {
                $data['supplier_location'] = '';
            }
        }

        $data['print_after_sale'] = $this->receiving_lib->is_print_after_sale();

        echo view("receivings/receiving", $data);
    }

    /**
     * @throws ReflectionException
     */
    public function postSave(int $receiving_id = -1): void    // TODO: Replace -1 with a constant
    {
        $newdate = $this->request->getPost('date', FILTER_SANITIZE_FULL_SPECIAL_CHARS);    // TODO: newdate does not follow naming conventions

        $date_formatter = date_create_from_format($this->config['dateformat'] . ' ' . $this->config['timeformat'], $newdate);
        $receiving_time = $date_formatter->format('Y-m-d H:i:s');

        $receiving_data = [
            'receiving_time' => $receiving_time,
            'supplier_id'    => $this->request->getPost('supplier_id') ? $this->request->getPost('supplier_id', FILTER_SANITIZE_NUMBER_INT) : null,
            'employee_id'    => $this->request->getPost('employee_id', FILTER_SANITIZE_NUMBER_INT),
            'comment'        => $this->request->getPost('comment', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
            'reference'      => $this->request->getPost('reference') != '' ? $this->request->getPost('reference', FILTER_SANITIZE_FULL_SPECIAL_CHARS) : null
        ];

        $this->inventory->update('RECV ' . $receiving_id, ['trans_date' => $receiving_time]);
        if ($this->receiving->update($receiving_id, $receiving_data)) {
            echo json_encode([
                'success' => true,
                'message' => lang('Receivings.successfully_updated'),
                'id'      => $receiving_id
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'message' => lang('Receivings.unsuccessfully_updated'),
                'id'      => $receiving_id
            ]);
        }
    }

    /**
     * Cancel an in-process receiving. Used in app/Views/receivings/receiving.php
     *
     * @return void
     * @noinspection PhpUnused
     */
    public function postCancelReceiving(): void
    {
        $this->receiving_lib->clear_all();

        $this->_reload();    // TODO: Hungarian Notation
    }
}
