<?php

namespace App\Controllers;

use App\Models\Branch_transfer;
use App\Models\Item;
use App\Models\Item_quantity;
use App\Models\Stock_location;
use Config\Services;

class Transfers extends Secure_Controller
{
    private Branch_transfer $branch_transfer;
    private Item $item;
    private Stock_location $stock_location;

    public function __construct()
    {
        parent::__construct('transfers');

        $this->branch_transfer = model(Branch_transfer::class);
        $this->item = model(Item::class);
        $this->stock_location = model(Stock_location::class);
    }

    /**
     * List all transfers.
     */
    public function getIndex(): void
    {
        $data['controller_name'] = 'transfers';
        $data['transfers'] = $this->branch_transfer->get_all(50, 0)->getResult();
        $data['stock_locations'] = $this->stock_location->get_allowed_locations('items');
        $data['config'] = config('OSPOS')->settings;
        $data['success'] = $this->session->getFlashdata('success');
        $data['error'] = $this->session->getFlashdata('error');

        echo view('transfers/manage', $data);
    }

    /**
     * Show new transfer form (for modal popup).
     */
    public function getAdd(): void
    {
        $locations = $this->stock_location->get_allowed_locations('items');
        if (count($locations) < 2) {
            echo '<div class="alert alert-warning">' . lang('Transfers.insufficient_stock') . ' ' . anchor('config', lang('Config.location_configuration')) . '</div>';
            return;
        }

        $data['stock_locations'] = $locations;
        $data['controller_name'] = 'transfers';

        echo view('transfers/form_content', $data);
    }

    /**
     * Process new transfer.
     */
    public function postAdd()
    {
        $fromLocationId = (int) $this->request->getPost('from_location_id');
        $toLocationId = (int) $this->request->getPost('to_location_id');
        $comment = $this->request->getPost('comment', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

        if ($fromLocationId === $toLocationId) {
            $this->session->setFlashdata('error', lang('Transfers.same_branch'));
            return redirect()->back()->withInput();
        }

        $items = [];
        $itemIds = $this->request->getPost('item_id');
        $qtyCartons = $this->request->getPost('qty_carton');
        $qtyPieces = $this->request->getPost('qty_piece');
        $qtyPers = $this->request->getPost('qty_per');
        if (is_array($itemIds)) {
            foreach ($itemIds as $i => $itemId) {
                $carton = (float) ($qtyCartons[$i] ?? 0);
                $piece = (float) ($qtyPieces[$i] ?? 0);
                $qtyPer = (float) ($qtyPers[$i] ?? 1);
                $qty = $qtyPer > 1 ? ($carton * $qtyPer + $piece) : $piece;
                if ($itemId && $qty > 0) {
                    $items[] = ['item_id' => $itemId, 'quantity' => $qty];
                }
            }
        }

        if (empty($items)) {
            $this->session->setFlashdata('error', lang('Transfers.no_items'));
            return redirect()->back()->withInput();
        }

        $employeeId = (int) $this->employee->get_logged_in_employee_info()->person_id;

        $transferId = $this->branch_transfer->create_transfer($fromLocationId, $toLocationId, $items, $employeeId, $comment);

        if ($transferId) {
            $this->session->setFlashdata('success', lang('Transfers.transfer_created'));
            return redirect()->to(site_url('transfers'));
        }
        $this->session->setFlashdata('error', lang('Transfers.transfer_failed'));
        return redirect()->back()->withInput();
    }

    /**
     * AJAX: Get item suggestions for transfer (items with stock at source).
     */
    public function getItemSearch(): void
    {
        $search = $this->request->getGet('term');
        $fromLocationId = (int) $this->request->getGet('from_location_id');
        if (!$fromLocationId) {
            echo json_encode([]);
            return;
        }

        $suggestions = $this->item->get_stock_search_suggestions($search, ['search_custom' => false, 'is_deleted' => false], true);
        $itemQuantity = model(Item_quantity::class);
        $result = [];
        foreach ($suggestions as $s) {
            $itemId = (int) ($s['value'] ?? $s['id'] ?? 0);
            if ($itemId > 0) {
                $qty = $itemQuantity->get_item_quantity($itemId, $fromLocationId);
                $available = (float) ($qty->quantity ?? 0);
                if ($available > 0) {
                    $s['available'] = $available;
                    $itemInfo = $this->item->get_info($itemId);
                    $qp = quantity_to_carton_pieces($available, $itemInfo);
                    $s['available_carton'] = $qp['carton'];
                    $s['available_pieces'] = $qp['pieces'];
                    $s['qty_per'] = $qp['qty_per'];
                    $s['bulk_label'] = $qp['bulk_label'];
                    $s['has_bulk'] = $qp['has_bulk'];
                    $result[] = $s;
                }
            }
        }
        echo json_encode($result);
    }

    /**
     * View transfer details.
     */
    public function getView(int $data_item_id = -1)
    {
        if ($data_item_id < 1) {
            return redirect()->to(site_url('transfers'));
        }
        $transfer = $this->branch_transfer->get_transfer_with_items($data_item_id);
        if (!$transfer) {
            return redirect()->to(site_url('transfers'));
        }
        $data['transfer'] = $transfer;
        $data['controller_name'] = 'transfers';
        echo view('transfers/view', $data);
    }
}
