<?php

namespace App\Database\Migrations;

use CodeIgniter\Database\Migration;

class Migration_ensure_employee_permissions extends Migration
{
    /**
     * Ensure default admin (person_id 1) and employees with module access have
     * required grants so they can work without errors. Fixes gaps where:
     * - Admin may be missing permissions after manual DB changes
     * - Employees with base module grants need at least one location when multiple exist
     */
    public function up(): void
    {
        $prefix = $this->db->getPrefix();
        $grantsTable = $prefix . 'grants';
        $permissionsTable = $prefix . 'permissions';
        $hasMenuGroup = $this->db->fieldExists('menu_group', $prefix . 'grants');

        // Ensure person 1 (default admin) has all base permissions
        $permissions = $this->db->table($permissionsTable)
            ->where('location_id IS NULL', null, false)
            ->get()
            ->getResultArray();

        $existingGrants = $this->db->table($grantsTable)
            ->where('person_id', 1)
            ->get()
            ->getResultArray();
        $existingPermIds = array_column($existingGrants, 'permission_id');

        $menuGroupMap = [];
        foreach ($existingGrants as $g) {
            $menuGroupMap[$g['permission_id']] = $g['menu_group'] ?? 'home';
        }

        foreach ($permissions as $perm) {
            $pid = $perm['permission_id'];
            if (in_array($pid, $existingPermIds, true)) {
                continue;
            }
            $data = ['permission_id' => $pid, 'person_id' => 1];
            if ($hasMenuGroup) {
                $data['menu_group'] = $menuGroupMap[$pid] ?? $this->defaultMenuGroup($pid);
            }
            $this->db->table($grantsTable)->insert($data);
        }

        // Ensure employees with items/sales/receivings but no location grant get first location
        // (avoids empty get_allowed_locations when they have only base grant)
        $locationCount = $this->db->table($prefix . 'stock_locations')->where('deleted', 0)->countAllResults();
        if ($locationCount < 2) {
            return;
        }

        $modules = ['items', 'sales', 'receivings'];
        foreach ($modules as $moduleId) {
            $locPerms = $this->db->table($permissionsTable)
                ->where('module_id', $moduleId)
                ->where('location_id IS NOT NULL', null, false)
                ->get()
                ->getResultArray();
            if (empty($locPerms)) {
                continue;
            }
            $firstLocPerm = $locPerms[0]['permission_id'];
            $peopleWithBase = $this->db->table($grantsTable)
                ->select('person_id')
                ->where('permission_id', $moduleId)
                ->get()
                ->getResultArray();
            foreach ($peopleWithBase as $row) {
                $personId = (int) $row['person_id'];
                $hasAnyLoc = $this->db->table($grantsTable)
                    ->where('person_id', $personId)
                    ->like('permission_id', $moduleId . '_', 'after')
                    ->countAllResults() > 0;
                if (!$hasAnyLoc) {
                    $menuGroup = $hasMenuGroup ? 'both' : null;
                    $data = ['permission_id' => $firstLocPerm, 'person_id' => $personId];
                    if ($menuGroup !== null) {
                        $data['menu_group'] = $menuGroup;
                    }
                    $this->db->table($grantsTable)->insert($data);
                }
            }
        }
    }

    private function defaultMenuGroup(string $permission_id): string
    {
        $office = ['config', 'employees', 'expenses_categories', 'taxes'];
        return in_array($permission_id, $office, true) ? 'office' : 'home';
    }

    public function down(): void
    {
        // No safe rollback - grants added by this migration are intentional
    }
}
