<?php

namespace App\Filters;

use App\Libraries\LicenseService;
use CodeIgniter\Filters\FilterInterface;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;

/**
 * Verifies GTS APPSTORE license before granting access to the application.
 * Redirects to license activation page when license is invalid, expired, or suspended.
 */
class LicenseFilter implements FilterInterface
{
    /**
     * Routes that do not require license verification.
     * Only license activation is excluded so users can enter their key.
     */
    private const EXCLUDED_ROUTES = [
        'license',
        'no_access',
    ];

    public function before(RequestInterface $request, $arguments = null)
    {
        $licenseConfig = config(\Config\License::class);
        // In production: always enforce license. In development: allow bypass when license.enabled = false
        if (ENVIRONMENT === 'production') {
            // Production: always run license check (ignore license.enabled to prevent accidental bypass)
        } elseif (! ($licenseConfig->enabled ?? true)) {
            return null; // Development/testing: bypass when disabled
        }

        $uri = $request->getUri();
        $path = trim($uri->getPath(), '/');
        $segments = $path ? explode('/', $path) : [];
        $firstSegment = $segments[0] ?? '';

        foreach (self::EXCLUDED_ROUTES as $excluded) {
            if ($firstSegment === $excluded || str_starts_with($path, $excluded . '/')) {
                return;
            }
        }

        $licenseService = new LicenseService();
        $result = $licenseService->verify();

        if ($result === null) {
            return $this->redirectToActivation('License verification failed.');
        }

        if (! ($result['valid'] ?? false) || ($result['status'] ?? '') !== 'active') {
            $message = $result['message'] ?? 'Your license is not active.';
            $status  = $result['status'] ?? 'invalid';

            return $this->redirectToActivation($message, $status);
        }

        return null;
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        return $response;
    }

    private function redirectToActivation(string $message, string $status = 'invalid')
    {
        return redirect()
            ->to(base_url('license/activate?message=' . rawurlencode($message) . '&status=' . rawurlencode($status)))
            ->with('license_error', $message)
            ->with('license_status', $status);
    }
}
