<?php

namespace App\Libraries;

use CodeIgniter\Database\BaseConnection;
use Config\Database;

/**
 * Database backup and reset operations for Configuration.
 */
class Database_lib
{
    private BaseConnection $db;
    private string $prefix;

    public function __construct()
    {
        $this->db   = Database::connect();
        $this->prefix = $this->db->getPrefix() ?: '';
    }

    /**
     * Feature groups for reset. Keys are feature ids, values are tables in truncate order (children before parents).
     */
    public function get_reset_features(): array
    {
        $p = $this->prefix;
        return [
            'sales'      => [
                'label'  => 'Sales',
                'tables' => [
                    $p . 'sales_payments',
                    $p . 'sales_items_taxes',
                    $p . 'sales_items',
                    $p . 'sales',
                    $p . 'sales_suspended_payments',
                    $p . 'sales_suspended_items_taxes',
                    $p . 'sales_suspended_items',
                    $p . 'sales_suspended',
                ],
            ],
            'receivings' => [
                'label'  => 'Receivings',
                'tables' => [$p . 'receivings_items', $p . 'receivings'],
            ],
            'items'      => [
                'label'  => 'Items & Inventory',
                'tables' => [
                    $p . 'attribute_links',
                    $p . 'inventory',
                    $p . 'item_quantities',
                    $p . 'item_kit_items',
                    $p . 'items_taxes',
                    $p . 'items',
                    $p . 'item_kits',
                ],
            ],
            'suppliers'  => [
                'label'  => 'Suppliers',
                'tables' => [$p . 'suppliers'],
                'pre_sql' => ["UPDATE `{$p}items` SET supplier_id = NULL"],
            ],
            'customers'  => [
                'label'  => 'Customers',
                'tables' => [$p . 'customers'],
                'pre_sql' => ["UPDATE `{$p}sales` SET customer_id = NULL", "UPDATE `{$p}sales_suspended` SET customer_id = NULL"],
            ],
            'transfers'  => [
                'label'  => 'Transfers',
                'tables' => [$p . 'branch_transfer_items', $p . 'branch_transfers'],
            ],
            'expenses'   => [
                'label'  => 'Expenses',
                'tables' => [$p . 'expenses'],
            ],
            'giftcards'  => [
                'label'  => 'Giftcards',
                'tables' => [$p . 'giftcards'],
            ],
        ];
    }

    /**
     * Reset a single feature by truncating its tables.
     */
    public function reset_feature(string $feature): bool
    {
        $features = $this->get_reset_features();
        if (!isset($features[$feature])) {
            return false;
        }

        $config = $features[$feature];
        $tables = $config['tables'];
        $preSql = $config['pre_sql'] ?? [];
        return $this->truncate_tables($tables, $preSql);
    }

    /**
     * Reset all transactional data. Keeps app_config, modules, permissions, grants, employees, people, stock_locations.
     */
    public function reset_all(): bool
    {
        $features = $this->get_reset_features();
        $allTables = [];
        $allPreSql = [];
        foreach ($features as $f) {
            foreach ($f['tables'] as $t) {
                $allTables[] = $t;
            }
            foreach ($f['pre_sql'] ?? [] as $sql) {
                $allPreSql[] = $sql;
            }
        }
        $allTables = array_unique($allTables);

        return $this->truncate_tables($allTables, array_unique($allPreSql));
    }

    /**
     * Truncate tables with FK checks disabled.
     *
     * @param array      $tables Tables to truncate
     * @param array|null $preSql Optional SQL to run before truncate (e.g. UPDATE to null FKs)
     */
    protected function truncate_tables(array $tables, ?array $preSql = null): bool
    {
        try {
            foreach ($preSql ?? [] as $sql) {
                try {
                    $this->db->query($sql);
                } catch (\Throwable $ignored) {
                    // Table might not exist
                }
            }

            $this->db->query('SET FOREIGN_KEY_CHECKS = 0');
            foreach ($tables as $table) {
                if ($this->db->tableExists($table)) {
                    $this->db->query("TRUNCATE TABLE `{$table}`");
                }
            }
            $this->db->query('SET FOREIGN_KEY_CHECKS = 1');
            return true;
        } catch (\Throwable $e) {
            $this->db->query('SET FOREIGN_KEY_CHECKS = 1');
            log_message('error', 'Database reset failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Create database backup using mysqldump. Returns path to backup file or false.
     */
    public function backup(): string|false
    {
        $config = config(Database::class)->default;
        $host   = $config['hostname'] ?? 'localhost';
        $user   = $config['username'] ?? 'root';
        $pass   = $config['password'] ?? '';
        $name   = $config['database'] ?? '';
        $port   = $config['port'] ?? 3306;

        $backupDir = WRITEPATH . 'backups';
        if (!is_dir($backupDir)) {
            mkdir($backupDir, 0755, true);
        }

        $filename = 'ospos_backup_' . date('Y-m-d_His') . '.sql';
        $filepath = $backupDir . DIRECTORY_SEPARATOR . $filename;

        $portArg = $port != 3306 ? " -P {$port}" : '';
        $passArg = $pass !== '' ? ' -p' . escapeshellarg($pass) : '';
        $mysqldump = $this->find_mysqldump();

        $cmd = sprintf(
            '%s -h %s -u %s%s%s %s > %s',
            escapeshellcmd($mysqldump),
            escapeshellarg($host),
            escapeshellarg($user),
            $passArg,
            $portArg,
            escapeshellarg($name),
            escapeshellarg($filepath)
        );

        exec($cmd . ' 2>&1', $output, $returnCode);
        if ($returnCode !== 0 || !file_exists($filepath)) {
            return false;
        }

        return $filepath;
    }

    /**
     * Find mysqldump executable (handles XAMPP/Windows common paths).
     */
    protected function find_mysqldump(): string
    {
        $paths = ['mysqldump'];
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            $xampp = getenv('XAMPP_HOME') ?: 'C:\\xampp';
            $paths[] = $xampp . '\\mysql\\bin\\mysqldump.exe';
        }
        foreach ($paths as $p) {
            $out = [];
            exec(escapeshellcmd($p) . ' --version 2>&1', $out, $ret);
            if ($ret === 0) {
                return $p;
            }
        }
        return 'mysqldump';
    }

    /**
     * Check if mysqldump is available.
     */
    public function is_backup_available(): bool
    {
        $out = [];
        exec(escapeshellcmd($this->find_mysqldump()) . ' --version 2>&1', $out, $ret);
        return $ret === 0;
    }
}
