<?php

namespace App\Libraries;

use App\Models\Appconfig;
use Config\License;
use Config\Services;

/**
 * GTS APPSTORE License verification service.
 * Verifies license keys via the GTS APPSTORE API and caches results.
 */
class LicenseService
{
    private const CACHE_KEY_PREFIX = 'gts_license_';
    private const CACHE_KEY_SUFFIX = '_verify';

    private Appconfig $appconfig;
    private License $config;

    public function __construct(?Appconfig $appconfig = null, ?License $config = null)
    {
        $this->appconfig = $appconfig ?? model(Appconfig::class);
        $this->config    = $config ?? config(License::class);
    }

    /**
     * Get the stored license key
     */
    public function getLicenseKey(): string
    {
        return $this->appconfig->get_value('gts_license_key', '');
    }

    /**
     * Get the stored secret token (optional, for extra security)
     */
    public function getSecretToken(): string
    {
        return $this->appconfig->get_value('gts_license_secret_token', '');
    }

    /**
     * Save license key and optionally secret token.
     *
     * @param array{license_key: string, secret_token?: string} $data
     */
    public function saveLicense(array $data): bool
    {
        $saved = true;
        if (! empty($data['license_key'])) {
            $saved = $this->appconfig->save(['gts_license_key' => trim($data['license_key'])]);
        }
        if ($saved && array_key_exists('secret_token', $data)) {
            $saved = $this->appconfig->save(['gts_license_secret_token' => trim($data['secret_token'] ?? '')]);
        }
        if ($saved) {
            $this->clearCache();
        }

        return $saved;
    }

    /**
     * Clear the verification cache (e.g. after license key change)
     */
    public function clearCache(): void
    {
        $cache = Services::cache();
        $key   = $this->getCacheKey();
        $cache->delete($key);
    }

    /**
     * Normalize domain: lowercase, no protocol, no trailing slash, no www unless that's the actual domain.
     */
    public function normalizeDomain(?string $host = null): string
    {
        $host = $host ?? ($_SERVER['HTTP_HOST'] ?? 'localhost');
        $host = strtolower(trim($host));
        if (str_starts_with($host, 'http://')) {
            $host = substr($host, 7);
        }
        if (str_starts_with($host, 'https://')) {
            $host = substr($host, 8);
        }
        $host = rtrim($host, '/');

        return $host;
    }

    /**
     * Verify the license. Uses cache if valid and not expired.
     * Returns verification result array or null on API/network error.
     *
     * @return array{status: string, message: string, expires_at?: string, in_grace_period?: bool, client_name?: string, application?: string, valid: bool}|null
     */
    public function verify(?string $domain = null, bool $forceRefresh = false): ?array
    {
        $licenseKey = $this->getLicenseKey();
        if (empty($licenseKey)) {
            return [
                'status'  => 'invalid',
                'message' => 'No license key configured.',
                'valid'   => false,
            ];
        }

        $domain = $this->normalizeDomain($domain);
        $cache  = Services::cache();
        $key    = $this->getCacheKey($domain);

        if (! $forceRefresh) {
            $cached = $cache->get($key);
            if ($cached !== null && is_array($cached)) {
                return $cached;
            }
        }

        $result = $this->callVerifyApi($licenseKey, $domain);
        if ($result !== null && $result['valid']) {
            $cache->save($key, $result, $this->config->cacheTtl);
        }

        return $result;
    }

    /**
     * Check if license is active (valid and status === 'active').
     */
    public function isActive(?string $domain = null): bool
    {
        $result = $this->verify($domain);

        return $result !== null
            && ($result['valid'] ?? false)
            && ($result['status'] ?? '') === 'active';
    }

    /**
     * Call the GTS APPSTORE verify API.
     *
     * @return array{status: string, message: string, expires_at?: string, in_grace_period?: bool, client_name?: string, application?: string, valid: bool}|null
     */
    private function callVerifyApi(string $licenseKey, string $domain): ?array
    {
        $client = Services::curlrequest(['timeout' => $this->config->timeout]);

        $payload = [
            'license_key' => $licenseKey,
            'domain'      => $domain,
            'app_id'      => $this->config->appId,
        ];
        $secretToken = $this->getSecretToken();
        if (! empty($secretToken)) {
            $payload['secret_token'] = $secretToken;
        }

        try {
            $response = $client->post($this->config->apiUrl, [
                'form_params' => $payload,
                'http_errors' => false,
            ]);
        } catch (\Throwable $e) {
            log_message('error', 'GTS License API error: ' . $e->getMessage());

            return $this->config->failOpen
                ? ['status' => 'active', 'message' => 'API unreachable (fail open)', 'valid' => true]
                : ['status' => 'invalid', 'message' => 'License verification service unavailable.', 'valid' => false];
        }

        $body = $response->getBody();
        $json = json_decode($body, true);
        if (! is_array($json)) {
            log_message('error', 'GTS License API invalid JSON: ' . substr($body, 0, 200));

            return $this->config->failOpen
                ? ['status' => 'active', 'message' => 'Invalid API response (fail open)', 'valid' => true]
                : ['status' => 'invalid', 'message' => 'Invalid license verification response.', 'valid' => false];
        }

        $data      = $json['data'] ?? [];
        $signature = $json['signature'] ?? '';
        $valid     = $json['valid'] ?? false;

        if (
            ! $this->config->skipSignatureVerification
            && ! empty($this->config->apiSecret)
            && ! empty($signature)
        ) {
            $toSign = json_encode($data);
            $expected = hash_hmac('sha256', $toSign, $this->config->apiSecret);
            if (! hash_equals($expected, $signature)) {
                // Try canonical encoding (sorted keys) in case API uses it
                if (is_array($data)) {
                    $this->sortRecursive($data);
                    $toSign   = json_encode($data);
                    $expected = hash_hmac('sha256', $toSign, $this->config->apiSecret);
                }
                if (! hash_equals($expected, $signature)) {
                    log_message('error', 'GTS License API signature verification failed');

                    return ['status' => 'invalid', 'message' => 'License response verification failed.', 'valid' => false];
                }
            }
        }

        return [
            'status'          => $data['status'] ?? 'invalid',
            'message'         => $data['message'] ?? 'Unknown',
            'expires_at'      => $data['expires_at'] ?? null,
            'in_grace_period' => $data['in_grace_period'] ?? false,
            'client_name'     => $data['client_name'] ?? null,
            'application'     => $data['application'] ?? null,
            'valid'           => $valid,
        ];
    }

    private function getCacheKey(?string $domain = null): string
    {
        $domain = $domain ?? $this->normalizeDomain();

        return self::CACHE_KEY_PREFIX . md5($domain . $this->getLicenseKey()) . self::CACHE_KEY_SUFFIX;
    }

    private function sortRecursive(array &$array): void
    {
        ksort($array);
        foreach ($array as &$value) {
            if (is_array($value)) {
                $this->sortRecursive($value);
            }
        }
    }
}
