<?php

namespace App\Models;

use CodeIgniter\Model;
use Config\Services;

/**
 * Dashboard model - provides summary data, analysis, and recommendations for the home page.
 */
class Dashboard extends Model
{
    protected $table = 'sales';
    protected $primaryKey = 'sale_id';

    /**
     * Get sales total for a date range (completed sales + invoices, excludes returns for simple total).
     */
    public function getSalesTotal(string $startDate, string $endDate): float
    {
        $builder = $this->db->table('sales_payments AS payments');
        $builder->join('sales AS sales', 'payments.sale_id = sales.sale_id', 'inner');
        $builder->select('COALESCE(SUM(payments.payment_amount), 0) AS total');
        $builder->where('DATE(sales.sale_time) >=', $startDate);
        $builder->where('DATE(sales.sale_time) <=', $endDate);
        $builder->where('sales.sale_status', COMPLETED);
        $builder->groupStart();
        $builder->whereIn('sales.sale_type', [SALE_TYPE_POS, SALE_TYPE_INVOICE, SALE_TYPE_RETURN]);
        $builder->groupEnd();

        $row = $builder->get()->getRow();
        return $row ? (float) $row->total : 0.0;
    }

    /**
     * Get transaction count for a date range.
     */
    public function getSalesCount(string $startDate, string $endDate): int
    {
        $builder = $this->db->table('sales');
        $builder->selectCount('sale_id', 'count');
        $builder->where('DATE(sale_time) >=', $startDate);
        $builder->where('DATE(sale_time) <=', $endDate);
        $builder->where('sale_status', COMPLETED);
        $builder->whereIn('sale_type', [SALE_TYPE_POS, SALE_TYPE_INVOICE, SALE_TYPE_RETURN]);

        $row = $builder->get()->getRow();
        return $row ? (int) $row->count : 0;
    }

    /** Cache TTL for credit totals (seconds) - reduces N+1 query load on dashboard */
    private const CREDIT_CACHE_TTL = 300;

    /**
     * Get total credit sales (debtors) - sum of all customers' outstanding balance.
     * Cached for 5 minutes to avoid N+1 queries on every dashboard load.
     */
    public function getTotalCreditSales(): float
    {
        $cache = Services::cache();
        $key   = 'dashboard_credit_sales';
        $total = $cache->get($key);
        if ($total !== null) {
            return (float) $total;
        }
        $customer = model(Customer::class);
        $total    = $customer->getTotalCreditBalance();
        $cache->save($key, $total, self::CREDIT_CACHE_TTL);
        return $total;
    }

    /**
     * Get total credit purchase (creditors) - sum of all suppliers' outstanding balance.
     * Cached for 5 minutes to avoid N+1 queries on every dashboard load.
     */
    public function getTotalCreditPurchase(): float
    {
        $cache = Services::cache();
        $key   = 'dashboard_credit_purchase';
        $total = $cache->get($key);
        if ($total !== null) {
            return (float) $total;
        }
        $supplier = model(Supplier::class);
        $total    = $supplier->getTotalCreditBalance();
        $cache->save($key, $total, self::CREDIT_CACHE_TTL);
        return $total;
    }

    /** Max items to show in dashboard debtors/creditors/inventory lists */
    private const DASHBOARD_LIST_LIMIT = 10;

    /**
     * Get list of customers with amount due (debtors). Sorted by balance desc.
     *
     * @return array<int, array{person_id: int, name: string, amount_due: float}>
     */
    public function getDebtorsList(): array
    {
        $customer = model(Customer::class);
        $customers = $customer->get_all(0, 0)->getResult();
        $list = [];
        foreach ($customers as $c) {
            $stats = $customer->get_credit_stats((int) $c->person_id);
            $balance = (float) $stats->credit_balance;
            if ($balance <= 0) {
                continue;
            }
            $name = trim($c->first_name . ' ' . $c->last_name);
            $name = $c->company_name ? $c->company_name . ($name ? " ({$name})" : '') : $name;
            $list[] = [
                'person_id'   => (int) $c->person_id,
                'name'        => $name ?: $c->company_name ?? '',
                'amount_due'  => $balance,
            ];
        }
        usort($list, fn($a, $b) => $b['amount_due'] <=> $a['amount_due']);

        return array_slice($list, 0, self::DASHBOARD_LIST_LIMIT);
    }

    /**
     * Get list of suppliers with credit balance due (creditors). Sorted by balance desc.
     *
     * @return array<int, array{person_id: int, name: string, credit_balance: float}>
     */
    public function getCreditorsList(): array
    {
        $supplier = model(Supplier::class);
        $suppliers = $supplier->get_all(0, 0)->getResult();
        $list = [];
        foreach ($suppliers as $s) {
            $stats = $supplier->get_credit_stats((int) $s->person_id);
            $balance = (float) $stats->credit_balance;
            if ($balance <= 0) {
                continue;
            }
            $name = trim($s->first_name . ' ' . $s->last_name);
            $name = $s->company_name ? $s->company_name . ($name ? " ({$name})" : '') : $name;
            $list[] = [
                'person_id'       => (int) $s->person_id,
                'name'            => $name ?: $s->company_name ?? '',
                'credit_balance'  => $balance,
            ];
        }
        usort($list, fn($a, $b) => $b['credit_balance'] <=> $a['credit_balance']);

        return array_slice($list, 0, self::DASHBOARD_LIST_LIMIT);
    }

    /**
     * Get list of items expired or expiring within 30 days.
     *
     * @return array<int, array{item_id: int, name: string, item_number: string, expire_date: string, quantity: float, location_name: string}>
     */
    public function getExpiringItemsList(): array
    {
        $itemsTable = $this->db->getPrefix() . 'items';
        if (!$this->db->fieldExists('expire_date', $itemsTable)) {
            return [];
        }
        $item = model(Item::class);
        $cutoff = date('Y-m-d', strtotime('+30 days'));

        $builder = $this->db->table('items AS items');
        $builder->select($item->get_item_name('name') . ', items.item_id, items.item_number, item_quantities.quantity, items.expire_date, stock_locations.location_name');
        $builder->join('item_quantities AS item_quantities', 'items.item_id = item_quantities.item_id');
        $builder->join('stock_locations', 'item_quantities.location_id = stock_locations.location_id');
        $builder->where('items.deleted', 0);
        $builder->where('items.stock_type', 0);
        $builder->where('items.expire_date IS NOT NULL');
        $builder->where('items.expire_date <=', $cutoff);
        $builder->where('stock_locations.deleted', 0);
        $builder->orderBy('items.expire_date', 'asc');
        $builder->limit(self::DASHBOARD_LIST_LIMIT);

        $rows = $builder->get()->getResultArray();
        return array_map(function ($row) {
            return [
                'item_id'       => (int) $row['item_id'],
                'name'          => $row['name'] ?? '',
                'item_number'   => $row['item_number'] ?? '',
                'expire_date'   => $row['expire_date'] ?? '',
                'quantity'      => (float) ($row['quantity'] ?? 0),
                'location_name' => $row['location_name'] ?? '',
            ];
        }, $rows);
    }

    /**
     * Get list of items out of stock (quantity <= 0).
     *
     * @return array<int, array{item_id: int, name: string, item_number: string, location_name: string}>
     */
    public function getOutOfStockItemsList(): array
    {
        $item = model(Item::class);

        $builder = $this->db->table('items AS items');
        $builder->select($item->get_item_name('name') . ', items.item_id, items.item_number, stock_locations.location_name');
        $builder->join('item_quantities AS item_quantities', 'items.item_id = item_quantities.item_id');
        $builder->join('stock_locations', 'item_quantities.location_id = stock_locations.location_id');
        $builder->where('items.deleted', 0);
        $builder->where('items.stock_type', 0);
        $builder->where('item_quantities.quantity <=', 0);
        $builder->where('stock_locations.deleted', 0);
        $builder->orderBy('items.name');
        $builder->limit(self::DASHBOARD_LIST_LIMIT);

        $rows = $builder->get()->getResultArray();
        return array_map(function ($row) {
            return [
                'item_id'       => (int) $row['item_id'],
                'name'          => $row['name'] ?? '',
                'item_number'   => $row['item_number'] ?? '',
                'location_name' => $row['location_name'] ?? '',
            ];
        }, $rows);
    }

    /**
     * Get receivings count for a date range.
     */
    public function getReceivingsCount(string $startDate, string $endDate): int
    {
        $builder = $this->db->table('receivings');
        $builder->selectCount('receiving_id', 'count');
        $builder->where('DATE(receiving_time) >=', $startDate);
        $builder->where('DATE(receiving_time) <=', $endDate);
        $row = $builder->get()->getRow();
        return $row ? (int) $row->count : 0;
    }

    /**
     * Get branch transfers count for a date range.
     */
    public function getTransfersCount(string $startDate, string $endDate): int
    {
        if (!$this->db->tableExists('branch_transfers')) {
            return 0;
        }
        $builder = $this->db->table('branch_transfers');
        $builder->selectCount('transfer_id', 'count');
        $builder->where('DATE(transfer_time) >=', $startDate);
        $builder->where('DATE(transfer_time) <=', $endDate);
        $row = $builder->get()->getRow();
        return $row ? (int) $row->count : 0;
    }

    /**
     * Get total items received (quantity) for a date range.
     */
    public function getItemsReceivedTotal(string $startDate, string $endDate): float
    {
        $builder = $this->db->table('receivings_items AS ri');
        $builder->select('COALESCE(SUM(ri.quantity_purchased * ri.receiving_quantity), 0) AS total');
        $builder->join('receivings AS r', 'ri.receiving_id = r.receiving_id');
        $builder->where('DATE(r.receiving_time) >=', $startDate);
        $builder->where('DATE(r.receiving_time) <=', $endDate);
        $row = $builder->get()->getRow();
        return $row ? (float) $row->total : 0.0;
    }

    /**
     * Get expenses total for a date range.
     */
    public function getExpensesTotal(string $startDate, string $endDate): float
    {
        if (!$this->db->tableExists('expenses')) {
            return 0.0;
        }
        $builder = $this->db->table('expenses');
        $builder->select('COALESCE(SUM(amount + IFNULL(tax_amount, 0)), 0) AS total');
        $builder->where('deleted', 0);
        $builder->where('DATE(date) >=', $startDate);
        $builder->where('DATE(date) <=', $endDate);
        $row = $builder->get()->getRow();
        return $row ? (float) $row->total : 0.0;
    }

    /**
     * Get count of items expiring within the next N days (or already expired).
     *
     * @param int $daysAhead Number of days to look ahead (default 30)
     * @return int
     */
    public function getExpiringSoonCount(int $daysAhead = 30): int
    {
        $itemsTable = $this->db->getPrefix() . 'items';
        if (!$this->db->fieldExists('expire_date', $itemsTable)) {
            return 0;
        }
        $cutoff = date('Y-m-d', strtotime("+{$daysAhead} days"));
        $builder = $this->db->table('items');
        $builder->selectCount('item_id', 'cnt');
        $builder->where('deleted', 0);
        $builder->where('stock_type', 0);
        $builder->where('expire_date IS NOT NULL');
        $builder->where('expire_date <=', $cutoff);
        $row = $builder->get()->getRow();
        return $row ? (int) $row->cnt : 0;
    }

    /**
     * Get low inventory items count.
     */
    public function getLowInventoryCount(): int
    {
        $items = $this->db->prefixTable('items');
        $iq = $this->db->prefixTable('item_quantities');
        $sl = $this->db->prefixTable('stock_locations');
        $sql = "SELECT COUNT(DISTINCT items.item_id) AS cnt
            FROM {$items} AS items
            JOIN {$iq} AS iq ON items.item_id = iq.item_id
            JOIN {$sl} AS sl ON iq.location_id = sl.location_id
            WHERE items.deleted = 0 AND items.stock_type = 0
            AND iq.quantity <= items.reorder_level
            AND sl.deleted = 0";
        $row = $this->db->query($sql)->getRow();
        return $row ? (int) $row->cnt : 0;
    }

    /**
     * Get daily sales data for charts (last N days).
     * Returns labels (formatted dates) and series (daily totals) for Chartist.
     *
     * @param int $days Number of days to include (default 7)
     * @return array{labels: string[], series: array}
     */
    public function getDailySalesChartData(int $days = 7): array
    {
        $endDate = date('Y-m-d');
        $startDate = date('Y-m-d', strtotime("-{$days} days"));

        $builder = $this->db->table('sales_payments AS payments');
        $builder->join('sales AS sales', 'payments.sale_id = sales.sale_id', 'inner');
        $builder->select('DATE(sales.sale_time) AS sale_date, COALESCE(SUM(payments.payment_amount), 0) AS total');
        $builder->where('DATE(sales.sale_time) >=', $startDate);
        $builder->where('DATE(sales.sale_time) <=', $endDate);
        $builder->where('sales.sale_status', COMPLETED);
        $builder->groupStart();
        $builder->whereIn('sales.sale_type', [SALE_TYPE_POS, SALE_TYPE_INVOICE, SALE_TYPE_RETURN]);
        $builder->groupEnd();
        $builder->groupBy('sale_date');
        $builder->orderBy('sale_date');

        $rows = $builder->get()->getResultArray();
        $byDate = [];
        foreach ($rows as $row) {
            $byDate[$row['sale_date']] = (float) $row['total'];
        }

        $labels = [];
        $series = [];
        for ($i = $days - 1; $i >= 0; $i--) {
            $d = date('Y-m-d', strtotime("-{$i} days"));
            $labels[] = to_date(strtotime($d));
            $series[] = ['meta' => to_date(strtotime($d)), 'value' => $byDate[$d] ?? 0];
        }

        return ['labels' => $labels, 'series' => $series];
    }

    /**
     * Get sales vs expenses for this month (for pie chart).
     *
     * @return array{labels: string[], series: array}
     */
    public function getSalesVsExpensesChartData(): array
    {
        $monthStart = date('Y-m-d', strtotime('first day of this month'));
        $today = date('Y-m-d');

        $salesTotal = $this->getSalesTotal($monthStart, $today);
        $expensesTotal = $this->getExpensesTotal($monthStart, $today);

        $labels = [lang('Dashboard.sales_chart'), lang('Dashboard.expenses_chart')];
        $series = [
            ['meta' => lang('Dashboard.sales_chart'), 'value' => $salesTotal],
            ['meta' => lang('Dashboard.expenses_chart'), 'value' => $expensesTotal],
        ];

        return ['labels' => $labels, 'series' => $series];
    }

    /**
     * Get dashboard summaries (today, week, month).
     *
     * @return array{summaries: array, analysis: array, recommendations: array}
     */
    public function getDashboardData(): array
    {
        $today = date('Y-m-d');
        $yesterday = date('Y-m-d', strtotime('-1 day'));
        $weekStart = date('Y-m-d', strtotime('monday this week'));
        $monthStart = date('Y-m-d', strtotime('first day of this month'));
        $lastWeekStart = date('Y-m-d', strtotime('monday last week'));
        $lastWeekEnd = date('Y-m-d', strtotime('sunday last week'));
        $lastMonthStart = date('Y-m-d', strtotime('first day of last month'));
        $lastMonthEnd = date('Y-m-d', strtotime('last day of last month'));

        $todayTotal = $this->getSalesTotal($today, $today);
        $todayCount = $this->getSalesCount($today, $today);
        $yesterdayTotal = $this->getSalesTotal($yesterday, $yesterday);
        $weekTotal = $this->getSalesTotal($weekStart, $today);
        $weekCount = $this->getSalesCount($weekStart, $today);
        $lastWeekTotal = $this->getSalesTotal($lastWeekStart, $lastWeekEnd);
        $monthTotal = $this->getSalesTotal($monthStart, $today);
        $monthCount = $this->getSalesCount($monthStart, $today);
        $lastMonthTotal = $this->getSalesTotal($lastMonthStart, $lastMonthEnd);

        $lowInventoryCount = $this->getLowInventoryCount();
        $creditSales = $this->getTotalCreditSales();
        $creditPurchase = $this->getTotalCreditPurchase();
        $todayReceivings = $this->getReceivingsCount($today, $today);
        $weekReceivings = $this->getReceivingsCount($weekStart, $today);
        $monthReceivings = $this->getReceivingsCount($monthStart, $today);
        $todayTransfers = $this->getTransfersCount($today, $today);
        $weekTransfers = $this->getTransfersCount($weekStart, $today);
        $monthTransfers = $this->getTransfersCount($monthStart, $today);
        $todayItemsReceived = $this->getItemsReceivedTotal($today, $today);
        $weekItemsReceived = $this->getItemsReceivedTotal($weekStart, $today);
        $monthItemsReceived = $this->getItemsReceivedTotal($monthStart, $today);
        $todayExpenses = $this->getExpensesTotal($today, $today);
        $weekExpenses = $this->getExpensesTotal($weekStart, $today);
        $monthExpenses = $this->getExpensesTotal($monthStart, $today);
        $expiringSoonCount = $this->getExpiringSoonCount(30);
        $avgSaleToday = $todayCount > 0 ? $todayTotal / $todayCount : 0;
        $avgSaleWeek = $weekCount > 0 ? $weekTotal / $weekCount : 0;
        $avgSaleMonth = $monthCount > 0 ? $monthTotal / $monthCount : 0;

        $summaries = [
            'today'  => ['total' => $todayTotal, 'count' => $todayCount, 'label' => 'today'],
            'week'   => ['total' => $weekTotal, 'count' => $weekCount, 'label' => 'this_week'],
            'month'  => ['total' => $monthTotal, 'count' => $monthCount, 'label' => 'this_month'],
        ];

        $todayVsYesterday = $yesterdayTotal > 0
            ? round((($todayTotal - $yesterdayTotal) / $yesterdayTotal) * 100, 1)
            : ($todayTotal > 0 ? 100 : 0);
        $weekVsLastWeek = $lastWeekTotal > 0
            ? round((($weekTotal - $lastWeekTotal) / $lastWeekTotal) * 100, 1)
            : ($weekTotal > 0 ? 100 : 0);
        $monthVsLastMonth = $lastMonthTotal > 0
            ? round((($monthTotal - $lastMonthTotal) / $lastMonthTotal) * 100, 1)
            : ($monthTotal > 0 ? 100 : 0);

        $analysis = [
            'today_trend'   => $todayVsYesterday,
            'week_trend'    => $weekVsLastWeek,
            'month_trend'   => $monthVsLastMonth,
        ];

        $recommendations = [];
        if ($lowInventoryCount > 0) {
            $recommendations[] = [
                'type'    => 'low_inventory',
                'count'   => $lowInventoryCount,
                'message' => 'low_inventory_recommendation',
            ];
        }
        if ($expiringSoonCount > 0) {
            $recommendations[] = [
                'type'    => 'expiring_soon',
                'count'   => $expiringSoonCount,
                'message' => 'expiring_soon_recommendation',
            ];
        }
        if ($todayTotal == 0 && (int) date('G') >= 9) {
            $recommendations[] = ['type' => 'no_sales_today', 'message' => 'no_sales_today_recommendation'];
        }
        if ($monthTotal > 0 && $monthVsLastMonth < -10) {
            $recommendations[] = ['type' => 'sales_decline', 'message' => 'sales_decline_recommendation'];
        }

        $salesChart = $this->getDailySalesChartData(7);
        $salesVsExpensesChart = $this->getSalesVsExpensesChartData();
        $debtors = $this->getDebtorsList();
        $creditors = $this->getCreditorsList();
        $expiringItems = $this->getExpiringItemsList();
        $outOfStockItems = $this->getOutOfStockItemsList();

        return [
            'summaries'              => $summaries,
            'analysis'               => $analysis,
            'recommendations'        => $recommendations,
            'credit_sales'           => $creditSales,
            'credit_purchase'        => $creditPurchase,
            'receivings'             => ['today' => $todayReceivings, 'week' => $weekReceivings, 'month' => $monthReceivings],
            'transfers'              => ['today' => $todayTransfers, 'week' => $weekTransfers, 'month' => $monthTransfers],
            'items_received'         => ['today' => $todayItemsReceived, 'week' => $weekItemsReceived, 'month' => $monthItemsReceived],
            'expenses'               => ['today' => $todayExpenses, 'week' => $weekExpenses, 'month' => $monthExpenses],
            'low_inventory'          => $lowInventoryCount,
            'expiring_soon'          => $expiringSoonCount,
            'average_sale'            => ['today' => $avgSaleToday, 'week' => $avgSaleWeek, 'month' => $avgSaleMonth],
            'sales_chart'            => $salesChart,
            'sales_vs_expenses_chart' => $salesVsExpensesChart,
            'debtors'                => $debtors,
            'creditors'              => $creditors,
            'expiring_items'         => $expiringItems,
            'out_of_stock_items'     => $outOfStockItems,
        ];
    }
}
