<?php

namespace App\Models\Reports;

use App\Models\Sale;

/**
 *
 *
 * @property sale sale
 *
 */
class Specific_supplier extends Report
{
    /**
     * @param array $inputs
     * @return void
     */
    public function create(array $inputs): void
    {
        // Create our temp tables to work with the data in our report
        $sale = model(Sale::class);
        $sale->create_temp_table($inputs);
    }

    /**
     * @return array[]
     */
    public function getDataColumns(): array
    {
        return [
            ['id'          => lang('Reports.sale_id')],
            ['type_code'   => lang('Reports.code_type')],
            ['sale_time'   => lang('Reports.date'), 'sortable' => false],
            ['name'        => lang('Reports.name')],
            ['category'    => lang('Reports.category')],
            ['item_number' => lang('Reports.item_number')],
            ['carton_qty'  => lang('Sales.carton_qty')],
            ['pieces_qty'  => lang('Sales.pieces_qty')],
            ['subtotal'    => lang('Reports.subtotal'), 'sorter' => 'number_sorter'],
            ['tax'         => lang('Reports.tax'), 'sorter' => 'number_sorter'],
            ['total'       => lang('Reports.total'), 'sorter' => 'number_sorter'],
            ['cost'        => lang('Reports.cost'), 'sorter' => 'number_sorter'],
            ['profit'      => lang('Reports.profit'), 'sorter' => 'number_sorter'],
            ['discount'    => lang('Reports.discount')]
        ];
    }

    /**
     * @param array $inputs
     * @return array
     */
    public function getData(array $inputs): array
    {
        $builder = $this->db->table('sales_items_temp AS sit');
        $builder->select('
            sit.sale_id,
            MAX(CASE
            WHEN sit.sale_type = ' . SALE_TYPE_POS . ' && sit.sale_status = ' . COMPLETED . ' THEN \'' . lang('Reports.code_pos') . '\'
            WHEN sit.sale_type = ' . SALE_TYPE_INVOICE . ' && sit.sale_status = ' . COMPLETED . ' THEN \'' . lang('Reports.code_invoice') . '\'
            WHEN sit.sale_type = ' . SALE_TYPE_WORK_ORDER . ' && sit.sale_status = ' . SUSPENDED . ' THEN \'' . lang('Reports.code_work_order') . '\'
            WHEN sit.sale_type = ' . SALE_TYPE_QUOTE . ' && sit.sale_status = ' . SUSPENDED . ' THEN \'' . lang('Reports.code_quote') . '\'
            WHEN sit.sale_type = ' . SALE_TYPE_RETURN . ' && sit.sale_status = ' . COMPLETED . ' THEN \'' . lang('Reports.code_return') . '\'
            WHEN sit.sale_status = ' . CANCELED . ' THEN \'' . lang('Reports.code_canceled') . '\'
            ELSE \'\'
            END) AS type_code,
            MAX(sit.sale_status) as sale_status,
            MAX(sit.sale_time) AS sale_time,
            MAX(sit.name) AS name,
            MAX(sit.category) AS category,
            MAX(sit.item_number) AS item_number,
            SUM(sit.quantity_purchased) AS items_purchased,
            SUM(sit.subtotal) AS subtotal,
            SUM(sit.tax) AS tax,
            SUM(sit.total) AS total,
            SUM(sit.cost) AS cost,
            SUM(sit.profit) AS profit,
            MAX(sit.discount_type) AS discount_type,
            MAX(sit.discount) AS discount');

        $builder->join('items AS report_items', 'sit.item_id = report_items.item_id', 'left');
        if ($this->db->fieldExists('item_unit', $this->db->getPrefix() . 'items')) {
            $builder->select('MAX(report_items.item_unit) AS item_unit, MAX(report_items.qty_per_unit) AS qty_per_unit, MAX(report_items.qty_per_carton) AS qty_per_carton, MAX(report_items.qty_per_box) AS qty_per_box, MAX(report_items.qty_per_bag) AS qty_per_bag, MAX(report_items.qty_per_pack) AS qty_per_pack, MAX(report_items.pack_name) AS pack_name');
        }

        $builder->where('sit.supplier_id', $inputs['supplier_id']);    // TODO: Duplicated code

        switch ($inputs['sale_type']) {
            case 'complete':
                $builder->where('sit.sale_status', COMPLETED);
                $builder->groupStart();
                $builder->where('sit.sale_type', SALE_TYPE_POS);
                $builder->orWhere('sit.sale_type', SALE_TYPE_INVOICE);
                $builder->orWhere('sit.sale_type', SALE_TYPE_RETURN);
                $builder->groupEnd();
                break;

            case 'sales':
                $builder->where('sit.sale_status', COMPLETED);
                $builder->groupStart();
                $builder->where('sit.sale_type', SALE_TYPE_POS);
                $builder->orWhere('sit.sale_type', SALE_TYPE_INVOICE);
                $builder->groupEnd();
                break;

            case 'quotes':
                $builder->where('sit.sale_status', SUSPENDED);
                $builder->where('sit.sale_type', SALE_TYPE_QUOTE);
                break;

            case 'work_orders':
                $builder->where('sit.sale_status', SUSPENDED);
                $builder->where('sit.sale_type', SALE_TYPE_WORK_ORDER);
                break;

            case 'canceled':
                $builder->where('sit.sale_status', CANCELED);
                break;

            case 'returns':
                $builder->where('sit.sale_status', COMPLETED);
                $builder->where('sit.sale_type', SALE_TYPE_RETURN);
                break;
        }

        $builder->groupBy(['sit.item_id', 'sit.sale_id']);
        $builder->orderBy('MAX(sit.sale_time)');

        return $builder->get()->getResultArray();
    }

    /**
     * @param array $inputs
     * @return array
     */
    public function getSummaryData(array $inputs): array
    {
        $builder = $this->db->table('sales_items_temp');
        $builder->select('SUM(subtotal) AS subtotal, SUM(tax) AS tax, SUM(total) AS total, SUM(cost) AS cost, SUM(profit) AS profit');
        $builder->where('supplier_id', $inputs['supplier_id']);

        switch ($inputs['sale_type']) {
            case 'complete':
                $builder->where('sale_status', COMPLETED);
                $builder->groupStart();
                $builder->where('sale_type', SALE_TYPE_POS);
                $builder->orWhere('sale_type', SALE_TYPE_INVOICE);
                $builder->orWhere('sale_type', SALE_TYPE_RETURN);
                $builder->groupEnd();
                break;

            case 'sales':
                $builder->where('sale_status', COMPLETED);
                $builder->groupStart();
                $builder->where('sale_type', SALE_TYPE_POS);
                $builder->orWhere('sale_type', SALE_TYPE_INVOICE);
                $builder->groupEnd();
                break;

            case 'quotes':
                $builder->where('sale_status', SUSPENDED);
                $builder->where('sale_type', SALE_TYPE_QUOTE);
                break;

            case 'work_orders':
                $builder->where('sale_status', SUSPENDED);
                $builder->where('sale_type', SALE_TYPE_WORK_ORDER);
                break;

            case 'canceled':
                $builder->where('sale_status', CANCELED);
                break;

            case 'returns':
                $builder->where('sale_status', COMPLETED);
                $builder->where('sale_type', SALE_TYPE_RETURN);
                break;
        }

        return $builder->get()->getRowArray();
    }
}
