#!/usr/bin/env php
<?php
/**
 * Build OSPOS GTS production deployment package.
 *
 * Creates a ZIP file with all files needed for cPanel or production deployment,
 * excluding development artifacts per .deployignore.
 *
 * Usage: php build_package.php [--optimize] [--no-clean]
 *   --optimize  Run composer install --no-dev --optimize-autoloader first
 *   --no-clean  Skip cleaning debugbar and cache
 */

$baseDir = realpath(__DIR__);
$options = [
    'optimize' => in_array('--optimize', $argv ?? [], true),
    'no-clean' => in_array('--no-clean', $argv ?? [], true),
];

// Load version from App config
$version = '3.4.1';
$appConfig = $baseDir . '/app/Config/App.php';
if (is_file($appConfig)) {
    $content = file_get_contents($appConfig);
    if (preg_match("/application_version\s*=\s*['\"]([^'\"]+)['\"]/", $content, $m)) {
        $version = $m[1];
    }
}

$distDir = $baseDir . '/dist';
$zipName = 'ospos_gts-' . $version . '-' . date('Ymd') . '.zip';
$zipPath = $distDir . '/' . $zipName;

// Create dist directory
if (!is_dir($distDir)) {
    mkdir($distDir, 0755, true);
}

echo "Building OSPOS GTS Production Package v{$version}\n";
echo str_repeat('-', 50) . "\n";

// Step 1: Clean
if (!$options['no-clean']) {
    echo "[1/3] Cleaning development artifacts...\n";
    $debugbarDir = $baseDir . '/writable/debugbar';
    if (is_dir($debugbarDir)) {
        foreach (glob($debugbarDir . '/*') ?: [] as $f) {
            if (is_file($f)) {
                unlink($f);
            }
        }
    }
    $cacheDir = $baseDir . '/writable/cache';
    if (is_dir($cacheDir)) {
        foreach (glob($cacheDir . '/*') ?: [] as $f) {
            if (is_file($f) && basename($f) !== 'index.html') {
                unlink($f);
            }
        }
    }
    echo "      Cleaned debugbar and cache.\n";
} else {
    echo "[1/3] Skipping clean (--no-clean).\n";
}

// Step 2: Composer optimize (optional)
if ($options['optimize']) {
    echo "[2/3] Running composer install --no-dev --optimize-autoloader...\n";
    $cmd = 'composer install --no-dev --optimize-autoloader';
    passthru($cmd);
} else {
    echo "[2/3] Skipping composer optimize (use --optimize to shrink vendor/).\n";
}

// Step 3: Create ZIP
echo "[3/3] Creating ZIP package...\n";

// Parse .deployignore
$ignorePath = $baseDir . '/.deployignore';
$excludePatterns = [];
if (is_file($ignorePath)) {
    $lines = file($ignorePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || str_starts_with($line, '#')) {
            continue;
        }
        if (str_starts_with($line, '!')) {
            continue; // Include overrides - we exclude by default, so skip
        }
        $excludePatterns[] = $line;
    }
}

// Additional excludes for build
$excludePatterns[] = 'dist/';
$excludePatterns[] = 'build_package.php';
$excludePatterns[] = 'prepare_deploy.php';

/**
 * Check if path should be excluded
 */
$isExcluded = function (string $relPath) use ($excludePatterns, $baseDir): bool {
    $path = str_replace('\\', '/', $relPath);
    foreach ($excludePatterns as $pat) {
        $pat = str_replace('\\', '/', trim($pat));
        // Directory match: .git/ matches .git/xxx
        if (str_ends_with($pat, '/')) {
            if (str_starts_with($path, $pat) || $path === rtrim($pat, '/')) {
                return true;
            }
        }
        // Exact match
        if ($path === $pat) {
            return true;
        }
        // Prefix match for dirs
        if (str_starts_with($path, $pat . '/')) {
            return true;
        }
        // Glob: *.log
        if (str_contains($pat, '*')) {
            $regex = '#^' . str_replace(['.', '*'], ['\.', '.*'], $pat) . '$#';
            if (preg_match($regex, $path)) {
                return true;
            }
            if (preg_match($regex, basename($path))) {
                return true;
            }
        }
    }
    return false;
};

$zip = new ZipArchive();
if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
    echo "ERROR: Cannot create ZIP file.\n";
    exit(1);
}

$fileCount = 0;
$iter = new RecursiveIteratorIterator(
    new RecursiveDirectoryIterator($baseDir, RecursiveDirectoryIterator::SKIP_DOTS | RecursiveDirectoryIterator::FOLLOW_SYMLINKS),
    RecursiveIteratorIterator::SELF_FIRST
);

foreach ($iter as $item) {
    $fullPath = $item->getPathname();
    $relPath = substr($fullPath, strlen($baseDir) + 1);
    $relPath = str_replace('\\', '/', $relPath);

    if ($isExcluded($relPath)) {
        continue;
    }

    if ($item->isDir()) {
        $zip->addEmptyDir($relPath);
    } else {
        $zip->addFile($fullPath, $relPath);
        $fileCount++;
    }
}

$zip->close();

$size = number_format(filesize($zipPath) / 1024 / 1024, 1);
echo "      Added {$fileCount} files.\n";
echo "\n" . str_repeat('-', 50) . "\n";
echo "Package created: dist/{$zipName}\n";
echo "Size: {$size} MB\n";
echo "\nNext steps:\n";
echo "1. Upload the ZIP to your cPanel hosting\n";
echo "2. Extract to your desired folder (e.g. /home/username/ospos_gts/)\n";
echo "3. Follow CPANEL_SETUP.md for setup or upgrade instructions\n";
echo "\n";
