# Reports Module – Recommendations for Better Reporting

This document summarizes findings from a review of the OSPOS reports codebase and provides actionable recommendations for improvement.

---

## 1. Architecture & Structure

### 1.1 Two Parallel Report Flows

There are two distinct report architectures that duplicate logic:

| Flow | Uses | Reports |
|------|------|---------|
| **Sale::create_temp_table()** | `sales_items_temp` temp table | Detailed_sales, Specific_*, Summary_sales (graphical) |
| **Summary_report::__common_select()** | Direct queries + its own temp tables | Summary_items, Summary_categories, Summary_customers, etc. |

**Recommendation:** Unify on a single temp-table creation flow. Either:

- Use `Sale::create_temp_table()` for all sale-based reports and retire `Summary_report`’s custom temp tables, or  
- Introduce a shared `SalesReportSchema` service that builds temp tables once per request and is used by both flows.

This reduces duplicated date filters, sale-type logic, and tax calculations.

### 1.2 Inconsistent Base Classes

- `Report` – minimal interface (`getDataColumns`, `getData`, `getSummaryData`).
- `Summary_report` – extended base with `_select`, `_from`, `_where`, `_group_order` template methods.
- Some reports (e.g. `Detailed_sales`) implement `Report` directly and bypass `Summary_report`.

**Recommendation:** Introduce an explicit hierarchy:

```
Report (abstract)
├── Summary_report (date range + sale type)
├── Detailed_report (summary + details + rewards)
├── Specific_report (entity-scoped: customer, supplier, employee)
└── Inventory_report
```

### 1.3 Controller Dependency on Every Report Model

`Reports` controller instantiates 16+ report models in the constructor, even for a single report run.

**Recommendation:** Use lazy loading or a factory:

```php
// Instead of constructor injection
private function getReport(string $name): Report {
    return model("App\Models\Reports\\" . $name);
}
```

---

## 2. Code Duplication

### 2.1 Sale Type Filter (Repeated 20+ times)

The same `switch ($inputs['sale_type'])` block appears in:

- Specific_customer, Specific_employee, Specific_supplier, Specific_discount
- Detailed_sales
- Summary_report::__common_where
- Sale::create_temp_table

**Recommendation:** Extract to a shared trait or helper:

```php
trait SaleTypeFilter {
    protected function applySaleTypeFilter($builder, string $saleType): void {
        switch ($saleType) {
            case 'complete': ...
            case 'sales': ...
            // etc.
        }
    }
}
```

### 2.2 Date/Time Filter (Repeated 10+ times)

Date filtering logic differs by `date_or_time_format` and appears in:

- Summary_report, Summary_expenses_categories, Summary_payments, Summary_sales_taxes
- Sale::create_temp_table

**Recommendation:** Centralize in a helper or service:

```php
function report_date_where(string $column, array $inputs): string {
    $config = config(OSPOS::class)->settings;
    return empty($config['date_or_time_format'])
        ? "DATE($column) BETWEEN ? AND ?"
        : "$column BETWEEN ? AND ?";
}
```

### 2.3 Location Filter

`if ($inputs['location_id'] != 'all')` is repeated in many report models.

**Recommendation:** Add `applyLocationFilter($builder, $locationId)` to the shared base or trait.

### 2.4 Tabular Row Mapping in Controller

Each report method in `Reports.php` has a similar pattern:

```php
$report_data = $this->summary_XXX->getData($inputs);
$tabular_data = [];
foreach ($report_data as $row) {
    $tabular_data[] = [
        'col1' => to_currency($row['x']),
        'col2' => to_quantity_decimals($row['y']),
        // ...
    ];
}
```

**Recommendation:** Have each report model define its own mapping:

```php
public function mapToTabularRow(array $row): array;
```

Then the controller can call:

```php
$tabular_data = array_map(fn($r) => $this->report->mapToTabularRow($r), $report_data);
```

Or introduce a `TabularReport` interface with `getDataColumns()` and `mapRow()`.

### 2.5 Detailed Report Details Loop

`detailed_sales`, `specific_customer`, `specific_employee`, `specific_discount`, `detailed_receivings` all use almost the same pattern:

- Load details per summary row
- Load rewards per summary row
- Call `report_quantity_carton_pieces()` for carton/pieces display

**Recommendation:** Add an abstract `Detailed_report` base that:

- Runs the summary query
- Iterates and loads details/rewards
- Applies `report_quantity_carton_pieces` in one place

---

## 3. Database & Schema Resilience

### 3.1 Schema Assumptions

Several reports assume columns that may not exist in older or customized databases:

- `sales_items.quantity_purchased` (some schemas use `quantity`)
- `sales_items.discount_type`
- `sales_items.discount`
- `items.item_unit`, `qty_per_carton`, etc.

**Recommendation:** Add a `ReportSchemaAdapter` that:

- Detects available columns (e.g. via `information_schema` or `getFieldNames()`)
- Exposes safe column names for use in SQL
- Is used by `Sale::create_temp_table`, `Detailed_sales`, `Summary_report`, etc.

### 3.2 Temp Table Aliasing

Some reports use `sales_items_temp` as a table alias, but CodeIgniter’s builder produces the prefixed name (e.g. `ospos_sales_items_temp`), so `sales_items_temp.name` can fail.

**Recommendation:** Either:

- Consistently use the prefixed table name when qualifying columns, or  
- Use `table('sales_items_temp AS sit')` and reference `sit.column_name`.

### 3.3 Duplicate Temp Table Creation

`Sale::create_temp_table` and `Summary_report::__common_select` both create:

- `sales_items_taxes_temp`
- `sales_payments_temp`

with slightly different definitions. This can lead to inconsistent results and wasted work.

**Recommendation:** Create these once per request (e.g. in a `SalesReportContext` service) and reuse them.

---

## 4. Controller Improvements

### 4.1 Method Signatures

Many report methods take 4–5 positional parameters:

```php
public function summary_sales(string $start_date, string $end_date, string $sale_type, string $location_id = 'all'): void
```

**Recommendation:** Use a single inputs array for all reports:

```php
public function summary_sales(array $inputs): void
// $inputs = ['start_date', 'end_date', 'sale_type', 'location_id', ...]
```

This simplifies routing and future parameter changes.

### 4.2 Repeated View Data Setup

Most report methods build data for the view in the same way:

```php
$data = [
    'title'        => lang('Reports.xxx'),
    'subtitle'     => $this->_get_subtitle_report($inputs),
    'headers'      => $this->report->getDataColumns(),
    'data'         => $tabular_data,
    'summary_data' => $summary
];
echo view('reports/tabular', $data);
```

**Recommendation:** Extract to a helper, e.g.:

```php
protected function renderTabularReport(Report $report, array $inputs, string $titleKey): void
```

### 4.3 Clear Cache

`$this->clearCache()` is called at the start of each report method without clear documentation of what is cached.

**Recommendation:** Document what is cached, and consider whether report data should be cacheable (e.g. with TTL) for performance.

---

## 5. Performance

### 5.1 N+1 Queries in Detailed Reports

`Detailed_sales`, `Specific_customer`, etc. run one query per summary row for details and rewards:

```php
foreach ($data['summary'] as $key => $value) {
    $data['details'][$key] = $builder->get()->getResultArray();  // N queries
    $data['rewards'][$key] = $builder->get()->getResultArray();  // N more
}
```

**Recommendation:** Batch load:

- Load all details with `sale_id IN (...)` and group in PHP
- Load all rewards with `sale_id IN (...)` and group in PHP

### 5.2 Temp Table Size

Temp tables can grow with large date ranges and many locations.

**Recommendation:** Add date range limits or pagination hints, and consider indexing strategies (e.g. on `sale_time`, `item_location`).

### 5.3 Unnecessary Model Instantiation

All report models are created in the controller constructor even when only one report is needed.

**Recommendation:** Use lazy loading or a factory so only the required report model is instantiated.

---

## 6. Maintainability

### 6.1 Hungarian Notation

TODOs in the code mention Hungarian notation (`_select`, `_from`, `_where`). The leading underscore suggests “internal” methods.

**Recommendation:** Pick a consistent style (e.g. `applySelect`, `buildFrom`) and document it in a coding standards file.

### 6.2 Magic Strings

Constants like `SALE_TYPE_POS`, `COMPLETED`, `PERCENT` are used, but some reports still use raw strings or `lang()` keys in queries.

**Recommendation:** Use constants everywhere and avoid embedding `lang()` in SQL (e.g. for `payment_type`).

### 6.3 fieldExists Usage

`fieldExists('item_unit', $this->db->getPrefix() . 'items')` is incorrect: `getPrefix()` returns `ospos_`, not a full table name. The second argument should be the full table name.

**Recommendation:** Use `$this->db->prefixTable('items')` and fix all `fieldExists` / `getFieldNames` calls accordingly.

---

## 7. UI/UX

### 7.1 Summary Display

`tabular.php` uses a hardcoded check for `total_quantity` vs currency formatting:

```php
if ($name == "total_quantity") {
    // quantity format
} else {
    // currency format
}
```

`tabular_details.php` improves this with `$summary_format`.

**Recommendation:** Generalize summary formatting (e.g. per-column formatters: `currency`, `quantity`, `integer`) so views do not need to special-case field names.

### 7.2 Export Options

Bootstrap Table export types are fixed (`json`, `xml`, `csv`, `txt`, `sql`, `excel`, `pdf`).

**Recommendation:** Make export options configurable and consider PDF layout for reports.

### 7.3 Error Handling

Database exceptions (e.g. missing columns) surface as raw CodeIgniter errors.

**Recommendation:** Add report-specific exception handling and user-friendly messages, e.g. “Report data could not be loaded. Please check that your database is up to date.”

---

## 8. Testing

### 8.1 Unit Tests

There are no obvious unit tests for report models.

**Recommendation:** Add tests for:

- `getData()` with known inputs
- `getSummaryData()` correctness
- Column existence handling (mocked schema)

### 8.2 Fixtures

Reports depend on a full DB schema and realistic data.

**Recommendation:** Add database fixtures (e.g. SQL files or migrations) for report tests to run against a known dataset.

---

## 9. Security

### 9.1 Input Validation

Report inputs (dates, IDs, sale types) are used in queries with limited validation.

**Recommendation:** Validate all report inputs (e.g. date format, allowed sale types, location IDs) before building queries.

### 9.2 SQL Injection

Most queries use the query builder or `escape()`, but some raw SQL in `Summary_report` and `Sale::create_temp_table` uses string concatenation.

**Recommendation:** Audit all raw SQL and ensure parameters are escaped or passed as bindings.

---

## 10. Quick Wins (Low Effort, High Impact)

| Priority | Action |
|----------|--------|
| 1 | Extract `applySaleTypeFilter()` into a shared trait – removes ~200 lines of duplication |
| 2 | Extract `buildDateWhere()` helper – centralizes date logic |
| 3 | Fix `fieldExists('item_unit', $this->db->getPrefix() . 'items')` → use `prefixTable('items')` |
| 4 | Add `ReportSchemaAdapter` for quantity/discount columns – prevents schema mismatch errors |
| 5 | Batch-load details in Detailed_sales (single query with `sale_id IN (...)`) |
| 6 | Use `report_items` consistently for item columns when joining – avoids ambiguous column errors |
| 7 | Add a `mapToTabularRow()` method to each report model – simplifies controller |

---

## 11. Suggested Implementation Order

1. **Phase 1 – Stabilization**
   - Fix schema detection (quantity column, discount columns)
   - Fix table aliasing for temp tables
   - Correct `fieldExists` usage

2. **Phase 2 – Deduplication**
   - Introduce `SaleTypeFilter` trait
   - Introduce date/location helpers
   - Unify tabular row mapping

3. **Phase 3 – Architecture**
   - Unify temp table creation
   - Introduce clear report base classes
   - Batch-load details in detailed reports

4. **Phase 4 – Quality**
   - Add unit tests for report models
   - Add input validation
   - Improve error handling and user-facing messages

---

---

## 12. Implemented (February 2026)

The following items from this document have been implemented:

- **SaleTypeFilter trait** – `app/Models/Reports/Traits/SaleTypeFilter.php` with `applySaleTypeFilter()` and `applyLocationFilter()`
- **Report helpers** – `report_date_where_clause()`, `report_items_have_item_unit()` in `app/Helpers/report_helper.php`
- **fieldExists fix** – Replaced `getPrefix() . 'items'` with `prefixTable('items')` or `report_items_have_item_unit()` where appropriate
- **ReportSchemaAdapter** – `app/Libraries/Report_schema.php` with `getSalesItemsQuantityColumn()` and `salesItemsHasDiscount()`
- **Refactored reports** – Specific_customer, Specific_employee, Specific_supplier, Specific_discount, Summary_report now use the trait and helpers
- **Batch loading** – Detailed_sales loads details and rewards in two batch queries instead of N+1 per sale

---

*Generated from codebase review – February 2026*
