<?php
/**
 * Live Server Diagnostic - Run this on your live server to identify issues.
 * Access via: https://yourdomain.com/health_check.php
 * DELETE this file after troubleshooting for security.
 */
header('Content-Type: text/plain; charset=utf-8');
$checks = [];

// PHP version
$checks['PHP Version'] = PHP_VERSION . ' (requires 8.1+)';

// Required extensions
$exts = ['intl', 'json', 'mbstring', 'mysqli', 'openssl'];
foreach ($exts as $ext) {
    $checks["Extension: $ext"] = extension_loaded($ext) ? 'OK' : 'MISSING';
}

// Path to .env
$root = dirname(__DIR__);
$envPath = $root . DIRECTORY_SEPARATOR . '.env';
$checks['.env exists'] = file_exists($envPath) ? 'OK' : 'MISSING';

// Writable directories
$writableDirs = ['writable', 'writable/cache', 'writable/logs', 'writable/session', 'writable/uploads'];
foreach ($writableDirs as $dir) {
    $path = $root . DIRECTORY_SEPARATOR . $dir;
    $exists = is_dir($path);
    $writable = $exists && is_writable($path);
    $checks["$dir"] = $exists ? ($writable ? 'OK (writable)' : 'NOT WRITABLE') : 'MISSING';
}

// Try to load .env and check key config
if (file_exists($envPath)) {
    $env = file_get_contents($envPath);
    $checks['CI_ENVIRONMENT'] = preg_match('/CI_ENVIRONMENT\s*=\s*(\w+)/', $env, $m) ? trim($m[1]) : 'not set';
    $checks['encryption.key'] = preg_match('/encryption\.key\s*=\s*(.+)/', $env, $m) && trim($m[1]) !== '' ? 'set' : 'MISSING or empty';
    if (preg_match('/database\.default\.hostname\s*=\s*(\S+)/', $env, $m)) $checks['DB host'] = trim($m[1]);
    if (preg_match('/database\.default\.database\s*=\s*(\S+)/', $env, $m)) $checks['DB name'] = trim($m[1]);
}

// Database connection test
$env = file_exists($envPath) ? file_get_contents($envPath) : '';
if ($env) {
    preg_match('/database\.default\.hostname\s*=\s*(\S+)/', $env, $h);
    preg_match('/database\.default\.database\s*=\s*(\S+)/', $env, $d);
    preg_match('/database\.default\.username\s*=\s*(\S+)/', $env, $u);
    preg_match('/database\.default\.password\s*=\s*(.*)/', $env, $p);
    $host = isset($h[1]) ? trim($h[1]) : 'localhost';
    $dbname = isset($d[1]) ? trim($d[1]) : '';
    $user = isset($u[1]) ? trim($u[1]) : 'root';
    $pass = isset($p[1]) ? trim($p[1]) : '';
    
    if ($dbname && extension_loaded('mysqli')) {
        $conn = @new mysqli($host, $user, $pass, $dbname);
        $checks['Database connection'] = $conn->connect_error ? 'FAILED: ' . $conn->connect_error : 'OK';
    }
}

// Output
echo "OSPOS Live Server Diagnostic\n";
echo str_repeat('=', 50) . "\n\n";
foreach ($checks as $label => $value) {
    echo sprintf("%-25s %s\n", $label . ':', $value);
}
echo "\nCommon fixes for live server:\n";
echo "- Database FAILED: Edit .env - hostname, database, username, password must match your live MySQL\n";
echo "- Writable NOT WRITABLE: chmod 755 or 775 on writable/ and all subfolders\n";
echo "- encryption.key MISSING: Run 'php spark key:generate' on live server, copy to .env\n";
echo "- To see actual error: Set CI_DEBUG=true in .env temporarily, refresh, then set back to false\n";
echo "- Subfolder install: Add app.baseURL = 'https://yourdomain.com/subfolder/public/' to .env\n";
echo "\n*** DELETE this file after troubleshooting! ***\n";
